# This example configures 4 I/Q stream channels using a shared LO, all
#   channels are phase coherent. Also shown is acquiring and plotting I/Q data. I/Q
#   data is provided as 32-bit floating point.

from pcrdevice.pcr_api import *

from scipy.signal import get_window
import matplotlib.pyplot as plt

# Number of active channels
CHANNELS = 4
# Must be a power of 2
DECIMATION = 1

def pcr_example_iq_multi_channel_coherent():
    # Connect the device
    status, handle = pcr_connect_device(PCR_DEFAULT_HOST_ADDR, PCR_DEFAULT_DEVICE_ADDR, PCR_DEFAULT_PORT).values()
    if status != PCRStatus.PCRStatusNoError:
        print(f'Error opening device: {pcr_get_error_string(status)}')
        return

    # Enable all channels
    ch_enabled = [False, False, False, False]
    for i in range(CHANNELS):
        ch_enabled[i] = True

    # Set channel configuration
    pcr_set_channel_config(handle, ch_enabled, PCR_SWEEP_CHANNEL_DISABLED)
    # All channels using shared LO
    for i in range(CHANNELS):
        pcr_set_channel_shared(handle, i, True)

    # Set shared LO frequency
    pcr_set_shared_freq(handle, 1.0e9)

    # Configure I/Q stream
    pcr_set_stream_data_type(handle, PCRDataType.PCRDataType32fc)
    pcr_set_stream_ref_level(handle, -20.0)
    pcr_set_stream_atten(handle, PCR_AUTO_ATTEN)
    pcr_set_stream_sample_rate(handle, DECIMATION)
    pcr_set_stream_bandwidth(handle, PCR_STREAM_MAX_BW / DECIMATION)
    # We are using the API to retrieve the I/Q samples, and not Vita49 streaming
    pcr_set_stream_mode(handle, PCRStreamMode.PCRStreamModeLocal)

    # Start the measurements
    pcr_initiate(handle)

    # Query the stream parameters
    status, channel_count, sample_rate, bandwidth = pcr_stream_parameters(handle).values()

    assert channel_count == CHANNELS
    assert sample_rate == (PCR_STREAM_SR / DECIMATION)

    # Number of samples to query at a time
    N = 1_000_000
    iq = np.zeros((CHANNELS, N)).astype(np.complex64)

    # Get I/Q
    pcr_stream_recv(handle, iq, N, False)

    # FFT and plot
    for ch in range(CHANNELS):
        # Create window
        window = get_window('hamming', len(iq[ch]))
        # Normalize window
        window *= len(window) / sum(window)
        # Window, FFT, normalize FFT output
        iq_data_FFT = np.fft.fftshift(np.fft.fft(iq[ch] * window) / len(window))
        # Convert to dBm
        plt.plot(10 * np.log10(iq_data_FFT.real ** 2 + iq_data_FFT.imag ** 2), label=f'Channel {ch}')

    plt.legend()
    plt.show()

    # Done, close device
    pcr_close_device(handle)


if __name__ == '__main__':
    pcr_example_iq_multi_channel_coherent()
