classdef PCRIQStreamer
    % This PCRIQStreamer object allows you to interface, configure, and
    % stream I/Q data from the PCR4200 4 channel spectrum analyzers.
    %
    % PCRIQStreamer methods:
    % PCRIQStreamer - Enumerate and establish connection with an PCR
    %   spectrum analyzer. Check the Status property to determine if a
    %   device connection was established. If this is the first time
    %   connecting to an PCR since it was power cycled, expect a
    %   60 second connection time, after which the connection time is
    %   reduced to roughly 10 seconds. Any active or previously opened
    %   devices found are first closed before attempting to establish
    %   connection. Because of this, only one device can be managed with
    %   this object, creating two objects does not result in two
    %   devices being managed.
    % delete - Closes the device through the device DLL. If the destructor
    %   is not called before attempting to reconnect to the same device,
    %   similar functionality is performed in the constructor. Therefore
    %   you only need to ensure the destructor is called if you wish to
    %   either reclaim the resources owned by the device DLL or you wish to
    %   interact with the device in another application/thread/etc.
    % samplerate - Retrieve the sample rate of the receiver based on the
    %   current value of the DecimationFactor property.
    % activeChannels - Retrieve all of the active channels for the select
    %   device
    % pair - Pair additional PCRs. You pass in the HostIP, DevIP and DevPort
    %   for the additional PCR. Only for use when you have systems using 
    %   more than a single PCR. 
    % start - Start the IQ data stream using the current value of the
    %   properties. Will stop an active IQ data stream if it wasn't
    %   previously stopped.
    % stop - Stop an active IQ data stream. Since the start() function
    %   will call stop() if the user has not done so, a main reason you
    %   might want to call stop is to reclaim resources owned by the device
    %   DLL (potentially several hundred MB of memory) and stop active
    %   processing which could be considerable based on the settings.
    % recv - Perform an IQ data acquisition on an active IQ data stream.
    %   Acquire 'n' complex samples from each active channel.
    %   If purge is set to true, discard any buffered leftover data.
    %   If purge is set to false retrieve the next contiguous 'n' samples.
    %   If continuous streaming is desired, this function should not be
    %   called with small sizes of 'n' as this function does contain some
    %   overhead. An acceptable size of 'n' might be 1/100th of the sample
    %   rate and above. For single block acquisitions, any size of 'n' is
    %   fine.
    % getstatusstring - Returns a string [1xN] char.
    %   Can be called after any class function to determing the success
    %   of the function.
    % 
    % PCRIQStreamer properties:
    % SerialNumber - Set by the constructor. If failed to initialize
    %   properly, will be set to -1.
    % NumDevices - Number of PCR devices connected together
    % HostAddr - IP Addreses used by the hosts. Up to 4 host IP addresses
    % DevAddr -  IP Addresses used by the PCR devices. Up to 4 device IP
    %   addresses
    % DevPort - The port used by the Devices. Up to 4. One for each
    % possible device.
    % ChannelsEnabled - A 4x4 array of integers representing 4 potential
    %   PCRs. 0 is disabled and 1 is enabled
    % IsChannelShared - A 4x4 array of bools representing 4 potential PCRs.
    %   True means that channel uses the shared LO and False means that
    %   channel uses the independent LO.
    % SharedCenterFrequency - Center frequency for the channels using the shared
    %   LO of the IQ data stream in Hz. Set directly.
    % IndepCenterFrequency - A 4x4 array of doubles representing the center
    %   frequency of each channel is using that channels Independent LO
    % DecimationFactor - Must be set to a power of two between [1, 4096].
    %   This value directly divides the base sample rate of the SM200 of
    %   50MS/s.
    % Bandwidth - User selectable bandwidth in Hz. When the decimation
    %   factor is greater than 8 or when the software filter is enabled,
    %   this value controls the applied digital filter bandwidth. It will
    %   be clamped to reasonable values for the given selected sample rate.
    %   Specified in Hz.
    % RefLevel - Set to any level above the expected input power (dBm).
    % OutputFormat - The IQ data is provided as single precision complex
    %   values. Set to 'interleaved' (fastest) to receive a single column
    %   vector where the values are interleaved complex pairs, [Re1; Im1;
    %   Re2; Im2; ...]. Set to 'non-interleaved' (slow) to retreive complex
    %   values as MATLAB defines them ([Nx2] matrix). Warning: the
    %   'non-interleaved' return format is significantly slower than the
    %   'interleaved' format and may have difficulties maintaining
    %   throughput at the highest sample rates.
    % Status - Reports the status of the last executed function. Use the
    %   getstatusstring() function to retrieve a human readable string of
    %   the reported status.
    
    properties
        % Internal use only.
        DeviceHandle = -1
        % Serial number of the device. Valid after construction.
        SerialNumber = -1
        % Number of devices
        NumDevices = 1
        % Host Address the first index is the primary device
        HostAddr = ['192.168.2.2'; '192.168.3.2'; '192.168.4.2'; '192.168.5.2']
        % PCR Device Address
        DevAddr = ['192.168.2.10'; '192.168.3.10'; '192.168.4.10'; '192.168.5.10']
        % Device Port 51665 is the default device port
        DevPort = [51665 51665 51665 51665]
        % Number of active channels
        ChannelsEnabled = [0 0 0 0; 0 0 0 0; 0 0 0 0; 0 0 0 0]
        % Are channels using shared LO
        IsChannelShared = [true true true true; true true true true; true true true true; true true true true]
        % Center frequency of the IQ data stream. (In Hz)
        SharedCenterFrequency = 1.0e9
        % Center frequency of the IQ data stream. (In Hz)
        IndepCenterFrequency = [1.0e9 1.0e9 1.0e9 1.0e9; 1.0e9 1.0e9 1.0e9 1.0e9; 1.0e9 1.0e9 1.0e9 1.0e9; 1.0e9 1.0e9 1.0e9 1.0e9]
        % Sample rate = 50.0e6 / DecimationFactor. Powers of 2 only.
        DecimationFactor = 1
        % IQ data bandwidth. (3dB point)
        Bandwidth = 40.0e6
        % Expected input power level. (dBm)
        RefLevel = 0.0
        % Specify the output IQ data format.
        OutputFormat = 'interleaved'
        % Last status reported by API.
        Status = 0
    end
    methods
        % Constructor
        % loads library, closes any open devices, and attempts to open a
        % new device. Specify either 'USB' or 'Networked' as the parameter.
        % Defaults to 'USB'. When 'Networked' uses the default network
        % configuration values.
        function obj = PCRIQStreamer()
            % Try to open the PCR API
            if not(libisloaded('pcr_api'))
                loadlibrary('pcr_api', 'pcr_api.h');
                % Test the library was loaded before continuing
                if(not(libisloaded('pcr_api')))
                    return;
                end
                
            else
                % Libary is already loaded. As a precaution, close all
                % possible handles It's possible the last execution ended
                % without properly closing the device, so just free all
                % resources and potentially opened devices. If at some
                % point you want to target multiple devices with this
                % class, you will need to either remove this logic or write
                % more targetted logic, i.e. close only the device you are
                % interested in, or maybe, look to see if a device is
                % already open and just return that?
                for i = (0 : 7)
                    calllib('pcr_api', 'pcrCloseDevice', i);
                end
            end
            
            % Create handle pointer
            handleptr = libpointer('int32Ptr', -1);
            
            % open networked device
            obj.Status = calllib('pcr_api', 'pcrConnectDevice', ...
                handleptr, obj.HostAddr(1, :), obj.DevAddr(1, :), obj.DevPort(1));

            if(obj.Status < 0)
                obj.DeviceHandle = -1;
                return;
            else
                obj.DeviceHandle = handleptr.value;
            end
            
            % Get serial number
            serialptr = libpointer('int32Ptr', -1);
            
            obj.Status = calllib('pcr_api', 'pcrGetSerialNumber', obj.DeviceHandle, serialptr);
            if(obj.Status >= 0)
                obj.SerialNumber = serialptr.value;
            end
        end % constructor
        
        function delete(obj)
            % Free resources associated with a device.
            calllib('pcr_api', 'pcrCloseDevice', obj.DeviceHandle);
        end
        
        function sr = samplerate(obj)
            % Retrieve the sample rate of the device. (In S/s)
            sr = 50.0e6 / obj.DecimationFactor;
        end

        function numChans = activeChannels(obj, dev)
            numChans = 0;
            for ch = 1:4
                if obj.ChannelsEnabled(dev, ch) ~= 0
                    numChans = numChans + 1;
                end
            end
        end

        function pair(obj)
            if obj.NumDevices > 1
                for dev = 2:obj.NumDevices
                    obj.Status = calllib('pcr_api', 'pcrPairDevice', ...
                    obj.DeviceHandle, obj.HostAddr(dev, :), obj.DevAddr(dev, :), obj.DevPort(dev));
                end
            end
        end
        
        function start(obj)
            % Initialize the IQ data stream.
            
            % Configuration
            
            flatChanEnabled = [obj.ChannelsEnabled(1, :) obj.ChannelsEnabled(2, :) obj.ChannelsEnabled(3, :) obj.ChannelsEnabled(4, :)];
            chEnPtr = libpointer('int32Ptr', flatChanEnabled);
            status = calllib('pcr_api', 'pcrSetChannelConfig', obj.DeviceHandle, chEnPtr, -1);
            str = calllib('pcr_api', 'pcrGetErrorString', status);
            fprintf('Status: %s\n', str);
            for dev = 1:obj.NumDevices
                channels = obj.activeChannels(dev);
                for ch = 1:channels
                    fprintf('Dev %d Chan %d IsShared %d IndFreq %f\n', dev, ch, int32(obj.IsChannelShared(dev,ch)), obj.IndepCenterFrequency(dev, ch));
                    status = calllib('pcr_api', 'pcrSetChannelShared', obj.DeviceHandle, ch-1, int32(obj.IsChannelShared(dev,ch)));
                    str = calllib('pcr_api', 'pcrGetErrorString', status);
                    fprintf('Status: %s\n', str);
                    status = calllib('pcr_api', 'pcrSetChannelFreq', obj.DeviceHandle, ch-1, obj.IndepCenterFrequency(dev, ch));
                    str = calllib('pcr_api', 'pcrGetErrorString', status);
                    fprintf('Status: %s\n', str);
                end
            end

            status = calllib('pcr_api', 'pcrSetSharedFreq', obj.DeviceHandle, obj.SharedCenterFrequency);
            str = calllib('pcr_api', 'pcrGetErrorString', status);
            fprintf('Status: %s\n', str);
            status = calllib('pcr_api', 'pcrSetVSGEnabled', obj.DeviceHandle, int32(false));
            str = calllib('pcr_api', 'pcrGetErrorString', status);
            fprintf('Status: %s\n', str);
            status = calllib('pcr_api', 'pcrSetStreamDataType', obj.DeviceHandle, 1); %32bit float
            str = calllib('pcr_api', 'pcrGetErrorString', status);
            fprintf('Status: %s\n', str);
            status = calllib('pcr_api', 'pcrSetStreamRefLevel', obj.DeviceHandle, obj.RefLevel);
            str = calllib('pcr_api', 'pcrGetErrorString', status);
            fprintf('Status: %s\n', str);
            status = calllib('pcr_api', 'pcrSetStreamAtten', obj.DeviceHandle, -1); %PCR Auto Atten
            str = calllib('pcr_api', 'pcrGetErrorString', status);
            fprintf('Status: %s\n', str);
            status = calllib('pcr_api', 'pcrSetStreamSampleRate', obj.DeviceHandle, obj.DecimationFactor);
            str = calllib('pcr_api', 'pcrGetErrorString', status);
            fprintf('Status: %s\n', str);
            status = calllib('pcr_api', 'pcrSetStreamBandwidth', obj.DeviceHandle, obj.Bandwidth);
            str = calllib('pcr_api', 'pcrGetErrorString', status);
            fprintf('Status: %s\n', str);
            status = calllib('pcr_api', 'pcrSetStreamMode', obj.DeviceHandle, int32(0));
            str = calllib('pcr_api', 'pcrGetErrorString', status);
            fprintf('Status: %s\n', str);
            
            % Start IQ
            status = calllib('pcr_api', 'pcrInitiate', obj.DeviceHandle);
            str = calllib('pcr_api', 'pcrGetErrorString', status);
            fprintf('Status: %s\n', str);

            chCntPtr = libpointer('int32Ptr');
            sampleratePtr = libpointer('doublePtr');
            bandwidthPtr = libpointer('doublePtr');

            status = calllib('pcr_api', 'pcrStreamParameters', obj.DeviceHandle, chCntPtr, sampleratePtr, bandwidthPtr);
            str = calllib('pcr_api', 'pcrGetErrorString', status);
            fprintf('Status: %s\n', str);
        end
        
        function stop(obj)
            % Halt the IQ data stream.
            calllib('pcr_api', 'pcrAbort', obj.DeviceHandle);
        end
        
        function iq = recv(obj, n, purge)
            totalActiveChans = 0;
            for dev = 1:obj.NumDevices
                totalActiveChans = totalActiveChans + obj.activeChannels(dev);
            end

            iqarrayptr = libpointer('singlePtr', zeros(n*2*totalActiveChans, 1));
            
            obj.Status = calllib('pcr_api', 'pcrStreamRecvFlat', obj.DeviceHandle, ...
                iqarrayptr, n, 0, int32(purge), 0, 0, 0, 0);
            if strcmp(obj.OutputFormat, 'non-interleaved')
                % Return results to MATLAB complex representation
                iq = zeros(n, totalActiveChans);
                start = 1;
                stop = n*2;
                for chan = 1:totalActiveChans
                    chData = iqarrayptr.Value(start:stop);
                    chComplexData = chData(1:2:end) + 1i*chData(2:2:end);
                    iq(:,chan) = chComplexData;
                    start = start + n*2;
                    stop = stop + n*2;
                end
            else
                % Return as interleaved I/Q pairs, column vector
                iq = iqarrayptr.Value;
            end
        end
        
        function str = getstatusstring(obj)
            % Returns a string [1xN] char.
            % Can be called after any class function to determing the success
            % of the function.
            str = calllib('pcr_api', 'pcrGetErrorString', obj.Status);
        end
    end % methods
end