#include "pcr_api.h"

// This example illustrates using the built in VSG to calibrate the phase
// offsets between channels. This example assumes the VSG output is split
// 4-way into all channels of the PCR.
// 
// A CW signal is fed into all 4 channels simultaneously, the PCR is configured
// for 4 channel phase synchronous I/Q streaming. Cross correlation is used
// to determine the phase offset between each channel. The calculated offset
// is applied through the API and a new measurement is made showing the 
// corrected offset.
// 
// The phase offset is calculated only at 1 frequency/configuration.
// If the frequency or temperature changes, a new calibration should occur.

#include <cassert>
#include <complex>
#include <iostream>
#include <vector>

static void checkStatus(PCRStatus sts)
{
    if(sts < PCRStatusNoError) {
        printf("Status error: %s\n", pcrGetErrorString(sts));
        assert(false);
    } else if(sts > PCRStatusNoError) {
        printf("Status warning: %s\n", pcrGetErrorString(sts));
    }
}

static double avgPowerDBM(std::vector<std::complex<float>> &src)
{
    double p = 0.0;
    for(int i = 0; i < src.size(); i++) {
        p += src[i].real() * src[i].real() + src[i].imag() * src[i].imag();
    }

    return 10.0 * log10(p / src.size());
}

static std::complex<float> cross_correlation(
    std::vector<std::complex<float>> &s1,
    std::vector<std::complex<float>> &s2)
{
    std::complex<float> result = 0;
    for(int i = 0; i < s1.size(); i++) {
        result += s1[i] * std::conj(s2[i]);
    }
    return result;
}

void pcr_example_phase_offset_adjust()
{
    PCRStatus sts;
    int handle;

    const int channels = 4;
    const float frequency = 1.0e9;
    const int decimation = 1; // 50MS/s

    // Connect the device
    sts = pcrConnectDevice(&handle, PCR_DEFAULT_HOST_ADDR, PCR_DEFAULT_DEVICE_ADDR, PCR_DEFAULT_PORT);
    if(sts != PCRStatusNoError) {
        printf("Error opening device: %s\n", pcrGetErrorString(sts));
        return;
    }

    int chEnabled[channels] = { 1, 1, 1, 1 };
    double avgPower[channels];
    double phaseOffset[channels] = { 0.0, 0.0, 0.0, 0.0 };

    // Enable all 4 channels for shared LO streaming
    // Initially, set all channels to zero phase offset
    sts = pcrSetChannelConfig(handle, chEnabled, PCR_SWEEP_CHANNEL_DISABLED);
    checkStatus(sts);
    for(int i = 0; i < channels; i++) {
        sts = pcrSetChannelShared(handle, i, PCRBoolTrue);
        checkStatus(sts);
    }

    // VSG uses the shared frequency. So this function will not only control
    //  any receive channels using the shared frequency, but also the VSG output frequency.
    // For this example, the recieve channel will use the shared frequency.
    sts = pcrSetSharedFreq(handle, frequency);
    checkStatus(sts);

    // Configure I/Q stream
    sts = pcrSetStreamDataType(handle, PCRDataType32fc);
    checkStatus(sts);
    sts = pcrSetStreamRefLevel(handle, -20.0);
    checkStatus(sts);
    sts = pcrSetStreamAtten(handle, PCR_AUTO_ATTEN);
    checkStatus(sts);
    sts = pcrSetStreamSampleRate(handle, decimation); 
    checkStatus(sts);
    sts = pcrSetStreamBandwidth(handle, 40.0e6);
    checkStatus(sts);
    // We are using the API to retrieve the I/Q samples, and not Vita49 streaming
    sts = pcrSetStreamMode(handle, PCRStreamModeLocal);
    checkStatus(sts);

    // Pattern is provided as interleaved I/Q samples
    // The pattern must be a multiple of 16 I/Q samples
    const int vsgPatternLen = 16;
    std::vector<std::complex<float>> vsgPattern(vsgPatternLen);
    for(int i = 0; i < vsgPatternLen; i++) {
        vsgPattern[i].real(1.0);
        vsgPattern[i].imag(0.0);
    }

    // Configure the VSG
    pcrSetVSGPattern(handle, vsgPattern.data(), vsgPatternLen, PCRDataType32fc);
    checkStatus(sts);
    pcrSetVSGLevel(handle, -25.0);
    checkStatus(sts);
    pcrSetVSGEnabled(handle, PCRBoolTrue);
    checkStatus(sts);

    // Start the measurements
    sts = pcrInitiate(handle);
    checkStatus(sts);

    // Query the stream parameters
    int channelCount;
    double sampleRate, bandwidth;
    sts = pcrStreamParameters(handle, &channelCount, &sampleRate, &bandwidth);
    checkStatus(sts);

    assert(channelCount == channels);
    assert(sampleRate == 50.0e6 / decimation);

    // Number of samples to query at a time
    int N = 1e6;
    void *ptr[channels];
    std::vector<std::complex<float>> iq[channels];

    for(int i = 0; i < channels; i++) {
        iq[i].resize(N);
        ptr[i] = &iq[i][0];
    }

    // Collect some data, throw away initial samples
    sts = pcrStreamRecv(handle, (void**)&ptr, N, 0, PCRBoolFalse, 0, 0, 0, 0);
    checkStatus(sts);
    sts = pcrStreamRecv(handle, (void**)&ptr, N, 0, PCRBoolFalse, 0, 0, 0, 0);
    checkStatus(sts);

    // Measure phase offsets between channels
    for(int ch = 0; ch < channels; ch++) {
        avgPower[ch] = avgPowerDBM(iq[ch]);
        std::complex<float> cc = cross_correlation(iq[0], iq[ch]);
        phaseOffset[ch] = -std::arg(cc);

        printf("Channel %d: Power, %.2f dBm, Phase Offset %.4f rad\n",
            ch+1, avgPower[ch], phaseOffset[ch]);
    }

    pcrAbort(handle);

    // Now apply the calculated phase offset
    for(int i = 0; i < channels; i++) {
        sts = pcrSetChannelPhaseOffset(handle, i, phaseOffset[i]);
        checkStatus(sts);
    }

    // Restart the stream and remeasure
    sts = pcrInitiate(handle);
    checkStatus(sts);

    // Collect some data, throw away initial samples
    sts = pcrStreamRecv(handle, (void**)&ptr, N, 0, PCRBoolFalse, 0, 0, 0, 0);
    checkStatus(sts);
    sts = pcrStreamRecv(handle, (void**)&ptr, N, 0, PCRBoolFalse, 0, 0, 0, 0);
    checkStatus(sts);

    // Measure phase offsets between channels
    for(int ch = 0; ch < channels; ch++) {
        avgPower[ch] = avgPowerDBM(iq[ch]);
        std::complex<float> cc = cross_correlation(iq[0], iq[ch]);
        phaseOffset[ch] = -std::arg(cc);

        printf("Channel %d: Power, %.2f dBm, Phase Offset %.4f rad\n",
            ch+1, avgPower[ch], phaseOffset[ch]);
    }

    pcrAbort(handle);

    // Done, close device
    pcrCloseDevice(handle);
}