#include "pcr_api.h"

// This example illustrates creating an 8 channel system using 2 PCR4200s.
// The process of connecting and configuring this system is illustrated.   
// Once configured, the 8 channel system is interfaced using a single handle
// and can be configured through all the standard API routines.

#include <cassert>
#include <complex>
#include <iostream>
#include <vector>

static void checkStatus(PCRStatus sts)
{
    if(sts < PCRStatusNoError) {
        printf("Status error: %s\n", pcrGetErrorString(sts));
        assert(false);
    } else if(sts > PCRStatusNoError) {
        printf("Status warning: %s\n", pcrGetErrorString(sts));
    }
}

static double avgPowerDBM(std::vector<std::complex<float>> &src)
{
    double p = 0.0;
    for(int i = 0; i < src.size(); i++) {
        p += src[i].real() * src[i].real() + src[i].imag() * src[i].imag();
    }

    return 10.0 * log10(p / src.size());
}

static std::complex<float> cross_correlation(
    std::vector<std::complex<float>> &s1,
    std::vector<std::complex<float>> &s2)
{
    std::complex<float> result = 0;
    for(int i = 0; i < s1.size(); i++) {
        result += s1[i] * std::conj(s2[i]);
    }
    return result;
}

void pcr_example_multi_receiver_2()
{
    PCRStatus sts;
    int handle;

    const char *deviceAddr1 = "192.168.2.10";
    const char *deviceAddr2 = "192.168.3.10";
    const char *hostAddr1 = "192.168.2.2";
    const char *hostAddr2 = "192.168.3.2";
    const uint16_t devicePort1 = PCR_DEFAULT_PORT;
    const uint16_t devicePort2 = PCR_DEFAULT_PORT;

    const int TOTAL_CHANNELS = 8;
    const double frequency = 1.0e9;
    const int decimation = 1;
    const double sampleRate = PCR_STREAM_SR / decimation;
    const double refLevel = -10.0;
    const double bandwidth = 40.0e6 / decimation;

    // Connect the primary device
    sts = pcrConnectDevice(&handle, hostAddr1, deviceAddr1, devicePort1);
    checkStatus(sts);
    // Pair the secondary device
    sts = pcrPairDevice(handle, hostAddr2, deviceAddr2, devicePort2);
    checkStatus(sts);

    // These functions set the local stream port. The streaming port is
    // different than the port used to connect the device. By default
    // the stream port will be 4991 on both devices. If on the same subnet
    // this will be a problem. This function allows you to change that.
    // In this example, the initial configuration was for 2 separate subnets,
    // and these functions are technically needed.
    sts = pcrSetStreamLocalAddr(handle, 0, PCR_DEFAULT_STREAM_PORT);
    checkStatus(sts);
    sts = pcrSetStreamLocalAddr(handle, 1, PCR_DEFAULT_STREAM_PORT+1);
    checkStatus(sts);

    // Confirm we have an 8 channel system
    int channelCount = 0;
    sts = pcrGetChannelCount(handle, &channelCount);
    checkStatus(sts);
    assert(channelCount == TOTAL_CHANNELS);

    int chEnabled[TOTAL_CHANNELS] = { 1, 1, 1, 1, 1, 1, 1, 1 };
    double avgPower[TOTAL_CHANNELS];
    double phaseOffset[TOTAL_CHANNELS];

    // Enable all 4 channels for shared LO streaming
    // Initially, set all channels to zero phase offset
    sts = pcrSetChannelConfig(handle, chEnabled, PCR_SWEEP_CHANNEL_DISABLED);
    checkStatus(sts);
    for(int i = 0; i < channelCount; i++) {
        sts = pcrSetChannelShared(handle, i, PCRBoolTrue);
        checkStatus(sts);
    }

    // VSG uses the shared frequency. So this function will not only control
    //  any receive channels using the shared frequency, but also the VSG output frequency.
    // For this example, the recieve channel will use the shared frequency.
    sts = pcrSetSharedFreq(handle, frequency);
    checkStatus(sts);

    // Configure I/Q stream
    sts = pcrSetStreamDataType(handle, PCRDataType32fc);
    checkStatus(sts);
    sts = pcrSetStreamRefLevel(handle, refLevel);
    checkStatus(sts);
    sts = pcrSetStreamAtten(handle, PCR_AUTO_ATTEN);
    checkStatus(sts);
    sts = pcrSetStreamSampleRate(handle, decimation);
    checkStatus(sts);
    sts = pcrSetStreamBandwidth(handle, bandwidth);
    checkStatus(sts);
    // We are using the API to retrieve the I/Q samples, and not Vita49 streaming
    sts = pcrSetStreamMode(handle, PCRStreamModeLocal);
    checkStatus(sts);

    // Pattern is provided as interleaved I/Q samples
    // The pattern must be a multiple of 16 I/Q samples
    // Fixed DC/CW signal
    const int vsgPatternLen = 16;
    std::vector<std::complex<float>> vsgPattern(vsgPatternLen);
    for(int i = 0; i < vsgPatternLen; i++) {
        vsgPattern[i].real(1.0);
        vsgPattern[i].imag(0.0);
    }

    // Configure the VSG
    pcrSetVSGPattern(handle, vsgPattern.data(), vsgPatternLen, PCRDataType32fc);
    checkStatus(sts);
    pcrSetVSGLevel(handle, refLevel);
    checkStatus(sts);
    pcrSetVSGEnabled(handle, PCRBoolTrue);
    checkStatus(sts);

    // Start the measurements
    sts = pcrInitiate(handle);
    checkStatus(sts);

    // Number of samples to query at a time
    int N = 1e6;
    void *ptr[TOTAL_CHANNELS];
    std::vector<std::complex<float>> iq[TOTAL_CHANNELS];

    for(int i = 0; i < TOTAL_CHANNELS; i++) {
        iq[i].resize(N);
        ptr[i] = &iq[i][0];
    }

    // Collect some data.
    for(int i = 0; i < 10; i++) {
        sts = pcrStreamRecv(handle, (void**)&ptr, N, 0, PCRBoolFalse, 0, 0, 0, 0);
        checkStatus(sts);
    }

    pcrAbort(handle);

    // Measure phase offsets between channels
    for(int ch = 0; ch < TOTAL_CHANNELS; ch++) {
        avgPower[ch] = avgPowerDBM(iq[ch]);
        std::complex<float> cc = cross_correlation(iq[0], iq[ch]);
        phaseOffset[ch] = -std::arg(cc);

        printf("Channel %d: Power, %.2f dBm, Phase Offset %.4f rad\n",
            ch+1, avgPower[ch], phaseOffset[ch]);
    }

    // Now apply the calculated phase offset
    for(int i = 0; i < TOTAL_CHANNELS; i++) {
        sts = pcrSetChannelPhaseOffset(handle, i, phaseOffset[i]);
        checkStatus(sts);
    }

    // Restart the stream and remeasure
    sts = pcrInitiate(handle);
    checkStatus(sts);

    // Collect some data.
    for(int i = 0; i < 10; i++) {
        sts = pcrStreamRecv(handle, (void**)&ptr, N, 0, PCRBoolFalse, 0, 0, 0, 0);
        checkStatus(sts);
    }

    pcrAbort(handle);

    // Measure phase offsets between channels
    for(int ch = 0; ch < TOTAL_CHANNELS; ch++) {
        avgPower[ch] = avgPowerDBM(iq[ch]);
        std::complex<float> cc = cross_correlation(iq[0], iq[ch]);
        phaseOffset[ch] = -std::arg(cc);

        printf("Channel %d: Power, %.2f dBm, Phase Offset %.4f rad\n",
            ch+1, avgPower[ch], phaseOffset[ch]);
    }

    // Done, close device
    pcrCloseDevice(handle);
}