#include "pcr_api.h"
#include <iostream>
#include <cassert>

// This example configures the PCR for 4 channel I/Q streaming, with
// each channel tuned to a different frequency using their indepdendent
// LOs. Also shown in this example is using the 16-bit integer data type
// for I/Q data, and how to convert this data to floating point.

void pcr_example_iq_multi_channel_independent()
{
    PCRStatus sts;
    int handle;

    // Connect the device
    sts = pcrConnectDevice(&handle, PCR_DEFAULT_HOST_ADDR, PCR_DEFAULT_DEVICE_ADDR, PCR_DEFAULT_PORT);
    if (sts != PCRStatusNoError) {
        printf("Error opening device: %s\n", pcrGetErrorString(sts));
        return;
    }

    const int CHANNELS = 4;
    int chEnabled[4] = { 1, 1, 1, 1 };
    int decimation = 1;
    int len = 1.0e6;

    // Allocate bufs for storing the IQ data
    int16_t* bufs[CHANNELS];
    float* bufsPwr[CHANNELS];
    float corrections[CHANNELS];
    for (int i = 0; i < CHANNELS; i++) {
        bufs[i] = new int16_t[len * 2]; // len * 2 because each IQ sample is constructed of 2 int16_t values
        bufsPwr[i] = new float[len];
    }

    // Configure Stream
    sts = pcrSetChannelConfig(handle, chEnabled, PCR_SWEEP_CHANNEL_DISABLED);
    assert(sts == PCRStatusNoError);
    for (int i = 0; i < CHANNELS; i++) {
        sts = pcrSetChannelShared(handle, i, PCRBoolFalse);
        assert(sts == PCRStatusNoError);
        double freqHz = (i + 1) * 1.0e9;
        sts = pcrSetChannelFreq(handle, i, freqHz); // Set each channel to a different frequency
        assert(sts == PCRStatusNoError);
    }

    sts = pcrSetVSGEnabled(handle, PCRBoolFalse);
    assert(sts == PCRStatusNoError);
    sts = pcrSetStreamDataType(handle, PCRDataType16sc);
    assert(sts == PCRStatusNoError);
    sts = pcrSetStreamRefLevel(handle, -20.0);
    assert(sts == PCRStatusNoError);
    sts = pcrSetStreamAtten(handle, PCR_AUTO_ATTEN);
    assert(sts == PCRStatusNoError);
    sts = pcrSetStreamSampleRate(handle, decimation);
    assert(sts == PCRStatusNoError);
    sts = pcrSetStreamBandwidth(handle, PCR_STREAM_MAX_BW / (double)decimation);
    assert(sts == PCRStatusNoError);

    sts = pcrInitiate(handle);
    assert(sts == PCRStatusNoError);

    // obtain corrections for each channel so we can properly scale the data
    for (int i = 0; i < CHANNELS; i++) {
        sts = pcrStreamCorrection(handle, i, &corrections[i]);
    }

    sts = pcrStreamRecv(handle, (void**)bufs, len, 0, PCRBoolTrue, 0, 0, 0, 0);
    assert(sts == PCRStatusNoError);

    
    for (int i = 0; i < CHANNELS; i++) {
        for (int j = 0; j < len; j++) {
            // Scale 16bit complex ints to floats from -1.0 to 1.0
            float re32f = ((float)bufs[i][j*2]) / 32768.0; 
            float im32f = ((float)bufs[i][j*2+1]) / 32768.0;

            // Apply amplitude corrections
            re32f *= corrections[i]; 
            im32f *= corrections[i]; 

            // Calculate Power
            bufsPwr[i][j] = 10.0 * log10(re32f * re32f + im32f * im32f);
        }
    }

    // Now do something with the signal power values you have obtained

    pcrCloseDevice(handle);
}