#include "pcr_api.h"

#include <iostream>
#include <cassert>
#include <vector>
#include <complex>

// This example configures 4 I/Q stream channels using a shared LO, all
//   channels are phase coherent. Also shown is acquiring I/Q data. I/Q
//   data is provided as 32-bit floating point.

static void checkStatus(PCRStatus sts)
{
    if(sts < PCRStatusNoError) {
        printf("Status error: %s\n", pcrGetErrorString(sts));
        assert(false);
    } else if(sts > PCRStatusNoError) {
        printf("Status warning: %s\n", pcrGetErrorString(sts));
    }
}

// Number of active channels
const int channels = 4;
// Must be a power of 2
const int decimation = 2;

void pcr_example_iq_multi_channel_coherent()
{
    PCRStatus sts;
    int handle;

    // Connect the device
    sts = pcrConnectDevice(&handle, PCR_DEFAULT_HOST_ADDR, PCR_DEFAULT_DEVICE_ADDR, PCR_DEFAULT_PORT);
    if(sts != PCRStatusNoError) {
        printf("Error opening device: %s\n", pcrGetErrorString(sts));
        return;
    }

    // Enable all channels
    int chEnabled[channels] = { 0, 0, 0, 0 };
    for(int i = 0; i < channels; i++) {
        chEnabled[i] = 1;
    }

    // Set channel configuration
    sts = pcrSetChannelConfig(handle, chEnabled, PCR_SWEEP_CHANNEL_DISABLED);
    checkStatus(sts);
    // All channels using shared LO
    for(int i = 0; i < 4; i++) {
        sts = pcrSetChannelShared(handle, i, PCRBoolTrue);
        checkStatus(sts);
    }
    // Set shared LO frequency
    sts = pcrSetSharedFreq(handle, 1.0e9);
    checkStatus(sts);

    // Configure I/Q stream
    sts = pcrSetStreamDataType(handle, PCRDataType32fc);
    checkStatus(sts);
    sts = pcrSetStreamRefLevel(handle, -20.0);
    checkStatus(sts);
    sts = pcrSetStreamAtten(handle, PCR_AUTO_ATTEN);
    checkStatus(sts);
    sts = pcrSetStreamSampleRate(handle, decimation);
    checkStatus(sts);
    sts = pcrSetStreamBandwidth(handle, PCR_STREAM_MAX_BW / decimation);
    checkStatus(sts);
    // We are using the API to retrieve the I/Q samples, and not Vita49 streaming
    sts = pcrSetStreamMode(handle, PCRStreamModeLocal);
    checkStatus(sts);

    // Start the measurements
    sts = pcrInitiate(handle);
    checkStatus(sts);

    // Query the stream parameters
    int channelCount;
    double sampleRate, bandwidth;
    sts = pcrStreamParameters(handle, &channelCount, &sampleRate, &bandwidth);
    checkStatus(sts);

    assert(channelCount == channels);
    assert(sampleRate == (PCR_STREAM_SR / decimation));

    // Number of samples to query at a time
    int N = 1e6;
    std::vector<std::complex<float>> iq[channels];
    void *bufs[channels];
    
    for(int i = 0; i < channels; i++) {
        iq[i].resize(N);
        bufs[i] = iq[i].data();
    }

    // Collect 1 second worth of data
    for(int i = 0; i < 25; i++) {
        sts = pcrStreamRecv(handle, bufs, N, 0, PCRBoolFalse, 0, 0, 0, 0);
        checkStatus(sts);

        // Do something with the data here.
        // The data is stored in 4 separate arrays that can be accessed through
        //   the iq vector variable.
    }

    // Done, close device
    pcrCloseDevice(handle);
}