import socket
from . import VRTPacket
from .vita49 import VITA49Decoder


class LiveStream:
    """
    Receives and decodes a Sceptre VITA-49 UDP LiveStream

    Usage:
        from sceptrelivestream import LiveStream, IFFrame

        livestream = LiveStream('', 12000)
        while True:
            pkt = livestream.read()
            if pkt is not None:
                if isinstance(pkt.frame, IFFrame):
                    # Process frame...
    """
    socket: socket.socket
    """Socket for UDP reception"""
    vita49: VITA49Decoder
    """VITA-49 decoder object"""

    def __init__(self, address: str = '', port: int = 12000, raw_output: bool = False):
        """
        Constructs a LiveStream object to listen on a UDP port

        :param address: Local address to bind to ('' for all interfaces)
        :param port: Local UDP port to listen on
        :param raw_output: Whether return data in the raw transmitted sample format.  If false, the data will be
                           converted to floating point and scaled based on VITA-49 gain metadata.  `raw_output = False`
                           is recommended since complex integer formats are not natively supported by NumPy and can be
                           difficult to work with.
        """
        self.socket = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
        self.socket.setsockopt(socket.SOL_SOCKET, socket.SO_RCVBUF, 16 * 1024 * 1024)
        self.socket.bind((address, port))
        self.vita49 = VITA49Decoder(raw_output)

    def read(self) -> VRTPacket:
        """
        Reads and decodes a single VRT packet from the network

        Each VRT packet will contain a VRT header.  If an associated context packet has been received for the VRT
        packet, it will be returned.  A decoded data payload (frame) may also be included in the resulting packet.

        :return: the VRT packet decoded from the network
        """
        while True:
            packet, addr = self.socket.recvfrom(65536)
            vrt_pkt = self.vita49.decode_packet(packet)
            if vrt_pkt is not None:
                return vrt_pkt
