// SACmdLineDemo
// main.cpp
// Demonstrates basic sweeps and I/Q streaming.


#include <iostream>
#include <cstdio>
#include <cstdlib>
#include <cmath>

#include "sa_api.h"

using namespace std;

// DoSweeps() Demonstrates basic sweeps. Reports any error codes encountered.
void DoSweeps(int device)
{
    // Configure the device
    cout << "saConfigAcquisition " << saGetErrorString(saConfigAcquisition(device, SA_MIN_MAX, SA_LOG_SCALE)) << "\n";
    cout << "saConfigCenterSpan " << saGetErrorString(saConfigCenterSpan(device, 915.0e6, 2.0e7)) << "\n";
    cout << "saConfigLevel " << saGetErrorString(saConfigLevel(device, -30.0)) << "\n";
    cout << "saConfigSweepCoupling " << saGetErrorString(saConfigSweepCoupling(device, 10.0e3, 10.0e3, false)) << "\n";

    // Enter Sweeping mode
    cout << "saInitiate " << saGetErrorString(saInitiate(device, SA_SWEEPING, 0)) << "\n";

    // Get some info about the sweep
    int sweepLen;
    double startFreq, binSize;
    saQuerySweepInfo(device, &sweepLen, &startFreq, &binSize);

    // Allocate arrays for the sweep data.
    float *min = new float[sweepLen];
    float *max = new float[sweepLen];

    // Get 10 sweeps
    for(int k=0; k<10; k++) {
      printf("Got Sweep %d\n", k);
      //        cout << "saGetSweep_32f " << saGetErrorString(saGetSweep_32f(device, min, max)) ;

        // Peak search
        int maxIdx = 0;
        for(int i=1; i<sweepLen; i++) {
            if(max[i] > max[maxIdx]) maxIdx = i;
        }

        // Display peak frequency, amplitude for each sweep.
        double peakFreq = 1.0e-6 * (startFreq + binSize * maxIdx);
        //cout << " Peak at " << peakFreq << " MHz, " << max[maxIdx] << " dBm" << "\n";
        usleep(10000);
    }

    // Stop sweeping and return the device to "idle"
    saAbort(device);

    //Avoid memory leaks
    delete [] min;
    delete [] max;

}

//StreamIQ: Inject a CW signal within 10 kHz of the center frequency, 0 to 30 dB below the reference level.
// This will stream about 1 second of I/Q data and look for phase discontinuities (indicating dropped data)
// Typically, decimation of 2 or 4 will have zero errors. Decimation of 1 occasionally has phase discontinuities.
void StreamIQ(int device)
{
    int decimation = 2; // Decimations of 1, 2, and 4 are available, with sample rates of (486.111 ksps / decimation)

    // Configure the device
    cout << "saConfigCenterSpan " << saGetErrorString(saConfigCenterSpan(device, 915.0e6, 2.0e5)) << "\n";
    cout << "saConfigLevel " << saGetErrorString(saConfigLevel(device, -30)) << "\n";
    cout << "saConfigIQ " << saGetErrorString(saConfigIQ(device, decimation, 10.0e3)) << "\n";

    // Set up for a chunk of data
    int numLoops = 30;
    int len;// = 16384 / decimation. May be obtained from saQueryStreamInfo.
    double bandwidth, sampleRate;

    // Initiate the I/Q streaming and get some info about the streaming
    cout << "saInitiate " << saGetErrorString(saInitiate(device, SA_IQ, 0)) << "\n";
    cout << "saQueryStreamInfo " << saGetErrorString(saQueryStreamInfo(device, &len, &bandwidth, &sampleRate)) << "\n";
    cout << "length " << len << "\n";

    // Allocate some memory for the streaming I/Q
    float * iqData = new float[len * numLoops * 2];

    // Get the data
    for(int i=0; i<numLoops; i++) {
         cout << "saGetIQ_32f " << saGetErrorString(saGetIQ_32f(device, &iqData[i * len * 2])) << "\n";
    }

    // Stop streaming
    saAbort(device);

    //Process the data.
    int phaseBreaks = 0;
    float lastPhase = atan2(iqData[1], iqData[0]);
    for(int i=1; i<len*numLoops; i++) {
        float thisPhase = atan2(iqData[i*2 + 1], iqData[i*2]);
        //Phase detect
        float deltaPhase = thisPhase - lastPhase;
        lastPhase = thisPhase;
        if(deltaPhase > 3.14159) deltaPhase -= 3.14159 * 2;
        if(deltaPhase < -3.14159) deltaPhase += 3.14159 * 2;
        //Count discontinuities. A CW tone at the center frequency should have no phase discontinuities unless data loss occurs.
        if((deltaPhase > 0.5) || (deltaPhase < -0.5)) {
            phaseBreaks++;
            if(phaseBreaks < 100) // Print up to 100 phase break locations
                cout << i << ": " << deltaPhase << " " << "\n";
        }

    }

    cout << phaseBreaks << " phase discontinuities detected." << "\n";
    delete [] iqData;
}

int main()
{
    int device;
    cout << "saOpenDevice " << saGetErrorString(saOpenDevice(&device)) << "\n";
    //cout << "device " << device << "\n";

    DoSweeps(device);
    StreamIQ(device);

    saCloseDevice(device);

    return 0;
}
