// ConstructSeqExample.cpp : This file contains the 'main' function. Program execution begins and ends there.
//

#include <iostream>
#include <fstream>
#include <vector>
using namespace std;

struct ArbSequenceHeader {
    uint32_t magic;
    uint32_t version;
};

enum ArbSequenceType {
    ArbSequenceTypeIQ,
    ArbSequenceTypeConstant,
    ArbSequenceTypeFreqChange,
};

struct c32 {
    float re, im;
};

struct ArbSequence {
public:
    ArbSequence() {
        type = ArbSequenceTypeIQ;
        id = 0;
        name = "";
        freq = 1.0e9;
        value.re = value.im = 0.0;
        counts = 0;
        repetitions = 1;
    }

    int IQLength() const;

    ArbSequenceType type;
    // unique number, used for GUI stuff, applies to all three types
    uint32_t id;
    // Used for GUI display only, applies to all three types
    string name;

    // Frequency step
    double freq;

    // Constant value
    c32 value;
    uint32_t counts;

    // Arb I/Q
    string fileName;
    vector<c32> arb;
    uint32_t repetitions;
};


const uint32_t ARB_SEQ_MAGIC_WORD = 0x51455356;

// Constants for markers in sequence file streams
// Markers are written out as 4 bytes
const uint32_t ARB_SEQ_CHUNK_IQ = 0xA0;
const uint32_t ARB_SEQ_CHUNK_CONSTANT = 0xA1;
const uint32_t ARB_SEQ_CHUNK_FREQ = 0xA2;

// Helper function for saving sequences files to disk
uint32_t WriteString(ofstream &f, const string& str) {
    const uint32_t length = str.length();
    f.write((char*)&length, 4);
    f.write(str.c_str(), length);
    return length;
}

int main()
{
    std::vector<ArbSequence*> sequencePackets; 

    // Set Center Frequency to 2.5GHz
    ArbSequence* seqFreq2_5GHz = new ArbSequence();
    seqFreq2_5GHz->type = ArbSequenceTypeFreqChange;
    seqFreq2_5GHz->name = "Set Freq 2.5GHz";
    seqFreq2_5GHz->freq = 2.5e9; // 2.5GHz
    sequencePackets.push_back(seqFreq2_5GHz);

    // Send IQ Data
    ArbSequence* seqIQFromFile = new ArbSequence();
    seqIQFromFile->type = ArbSequenceTypeIQ;
    seqIQFromFile->name = "Random IQ Sequence";
    seqIQFromFile->fileName = "madeup.bin";
    for(int i = 0; i < 1000; i++) {
        c32 temp;
        temp.re = 1 * i / 1000.0;
        temp.im = 0;
        seqIQFromFile->arb.push_back(temp);
    }
    sequencePackets.push_back(seqIQFromFile);

    // Set Center Frequency to 1.0GHz
    ArbSequence* seqFreq1GHz = new ArbSequence();
    seqFreq1GHz->type = ArbSequenceTypeFreqChange;
    seqFreq1GHz->name = "Set Freq 1.0GHz";
    seqFreq1GHz->freq = 1.0e9; // 1.0 GHz
    sequencePackets.push_back(seqFreq1GHz);

    // Send constants cw
    ArbSequence* seqConstants = new ArbSequence();
    seqConstants->type = ArbSequenceTypeConstant;
    seqConstants->name = "Constants";
    c32 temp;
    temp.re = 1.0;
    temp.im = 0.0;
    seqConstants->counts = 1000;
    seqConstants->value = temp;
    sequencePackets.push_back(seqConstants);

    // Write Sequence File
    ofstream f("example.seq", ios::out | ios::binary | ios::ate);

    // Write out header
    ArbSequenceHeader hdr;
    hdr.magic = ARB_SEQ_MAGIC_WORD;
    hdr.version = 1;
    f.write((char*)&hdr, sizeof(hdr));

    for(int i = 0; i < sequencePackets.size(); i++) {
        if(sequencePackets[i]->type == ArbSequenceTypeIQ) {
            f.write((char*)&ARB_SEQ_CHUNK_IQ, 4);
            WriteString(f, sequencePackets[i]->name);
            WriteString(f, sequencePackets[i]->fileName);
            uint32_t samples = sequencePackets[i]->arb.size();
            f.write((char*)&samples, 4);
            f.write((char*)&sequencePackets[i]->repetitions, 4);
            //TODO: convert to shorts
            f.write((char*)sequencePackets[i]->arb.data(), sequencePackets[i]->arb.size() * 8);
        } else if(sequencePackets[i]->type == ArbSequenceTypeConstant) {
            f.write((char*)&ARB_SEQ_CHUNK_CONSTANT, sizeof(ARB_SEQ_CHUNK_CONSTANT));
            WriteString(f, sequencePackets[i]->name);
            f.write((char*)&sequencePackets[i]->value, 8);
            f.write((char*)&sequencePackets[i]->counts, 4);
        } else if(sequencePackets[i]->type == ArbSequenceTypeFreqChange) {
            f.write((char*)&ARB_SEQ_CHUNK_FREQ, sizeof(ARB_SEQ_CHUNK_FREQ));
            WriteString(f, sequencePackets[i]->name);
            f.write((char*)&sequencePackets[i]->freq, 8);
        } else {
            // Should never get here
        }
    }
}

// Run program: Ctrl + F5 or Debug > Start Without Debugging menu
// Debug program: F5 or Debug > Start Debugging menu

// Tips for Getting Started: 
//   1. Use the Solution Explorer window to add/manage files
//   2. Use the Team Explorer window to connect to source control
//   3. Use the Output window to see build output and other messages
//   4. Use the Error List window to view errors
//   5. Go to Project > Add New Item to create new code files, or Project > Add Existing Item to add existing code files to the project
//   6. In the future, to open this project again, go to File > Open > Project and select the .sln file
