// Copyright (c).2025, Signal Hound, Inc.
// For licensing information, please see the API license in the software_licenses folder

/*!
 * \file pcr_api.h
 * \brief API functions for the PCR4200 spectrum analyzers.
 *
 * This is the main file for user accessible functions for controlling the
 * PCR4200 reciever.
 */

#ifndef PCR_API_H
#define PCR_API_H

#if defined(_WIN32) // Windows
    #ifdef PCR_EXPORTS
        #define PCR_API __declspec(dllexport)
    #else
        #define PCR_API
    #endif

    // bare minimum stdint typedef support
    #if _MSC_VER < 1700 // For VS2010 or earlier
        typedef signed char        int8_t;
        typedef short              int16_t;
        typedef int                int32_t;
        typedef long long          int64_t;
        typedef unsigned char      uint8_t;
        typedef unsigned short     uint16_t;
        typedef unsigned int       uint32_t;
        typedef unsigned long long uint64_t;
    #else
        #include <stdint.h>
    #endif

    #define PCR_DEPRECATED(comment) __declspec(deprecated(comment))
#else // Linux
    #include <stdint.h>
    #define PCR_API __attribute__((visibility("default")))

    #if defined(__GNUC__)
        #define PCR_DEPRECATED(comment) __attribute__((deprecated))
    #else
        #define PCR_DEPRECATED(comment) comment
    #endif
#endif

/** Maximum number of device that can be interfaced by the API */
#define PCR_MAX_DEVICES (4)

/** Convenience host address */
#define PCR_DEFAULT_HOST_ADDR ("192.168.2.2")
/** Default PCR IP address */
#define PCR_DEFAULT_DEVICE_ADDR ("192.168.2.10")
/** Default PCR port */
#define PCR_DEFAULT_PORT (51665)
/** Default streaming port */
#define PCR_DEFAULT_STREAM_PORT (4991)

/** Used in pcrSetChannelConfig to indicate no sweep channel */
#define PCR_SWEEP_CHANNEL_DISABLED (-1)

/** Indicates the API should use the reference level to control sensitivity. Default */
#define PCR_AUTO_ATTEN (-1)
/** Valid attenuation values between [0, 6], or -1 for auto */
#define PCR_MAX_ATTEN (6)

/** Minimum configurable frequency */
#define PCR_MIN_FREQ (100.0e3)
/** Maximum configurable frequency */
#define PCR_MAX_FREQ (20.0e9)
/** Minimum configurable sweep span */
#define PCR_SWEEP_MIN_SPAN (100.0)
/** Minimum configurable sweep RBW */
#define PCR_SWEEP_MIN_RBW (100.0)
/** Maximum configurable sweep RBW */
#define PCR_SWEEP_MAX_RBW (10.0e6)
/** Minimum configurable sweep time */
#define PCR_MIN_SWEEP_TIME (1.0e-6)
/** Maximum configurable sweep time */
#define PCR_MAX_SWEEP_TIME (1.0)

/** Minimum configurable reference level, for sweeps and I/Q streaming */
#define PCR_MIN_REF_LVL (-130.0)
/** Maximum configurable reference level, for sweeps and I/Q streaming */
#define PCR_MAX_REF_LVL (20.0)

/** The maximum I/Q stream sample rate, decimation = 1 */
#define PCR_STREAM_SR (50.0e6)
/** Minimum configurable I/Q stream decimation */
#define PCR_STREAM_MIN_DECIMATION (1)
/** Maximum configurable I/Q stream decimation */
#define PCR_STREAM_MAX_DECIMATION (256)
/** Minimum configurable I/Q stream bandwidth */
#define PCR_STREAM_MIN_BW (20.0e3)
/** Maximum configurable I/Q stream bandwidth */
#define PCR_STREAM_MAX_BW (40.0e6)

/** Minimum Vita49 stream mode context interval */
#define PCR_MIN_CNTX_INTERVAL (3)
/** Maximum Vita49 stream mode context interval */
#define PCR_MAX_CNTX_INTERVAL (4095)

/** Minimum VSG output level (dBm) */
#define PCR_VSG_MIN_LEVEL (-80.0)
/** Maximum VSG output level (dBm) */
#define PCR_VSG_MAX_LEVEL (10.0)
/** Maximum VSG pattern buffer length (complex samples) */
#define PCR_VSG_MAX_BUF_LEN (4096)

/** Minimum fan setpoint in C */
#define PCR_MIN_FAN_SETPOINT (5.0)
/** Maximum fan setpoint in C */
#define PCR_MAX_FAN_SETPOINT (60.0)

/** Minimum channel delay adjustment in seconds */
#define PCR_MIN_CHANNEL_DELAY (-25.0e-9)
/** Maximum channel delay adjustment in seconds */
#define PCR_MAX_CHANNEL_DELAY (25.0e-9)

/**
 * Boolean data type
 */
typedef enum PCRBool {
    /** False */
    PCRBoolFalse = 0,
    /** True */
    PCRBoolTrue = 1
} PCRBool;

/**
 * Controls the I/Q data type returned from the #pcrStreamRecv function.
 */
typedef enum PCRDataType {
    /** 16-bit interleaved complex integers. */
    PCRDataType16sc = 0,
    /** 32-bit interleaved complex floating point. */
    PCRDataType32fc = 1
} PCRDataType;

/**
 * Controls the timebase reference clock of the system.
 */
typedef enum PCRReference {
    /** Use the internal 10MHz reference clock. */
    PCRReferenceInternal = 0,
    /** Use 10MHz reference connected to the reference input RF port. */
    PCRReferenceExternal = 1
} PCRReference;

/**
 * Specifies the current state of GPS lock.
 */
typedef enum PCRGPSState {
    /** No GPS lock detected. */
    PCRGPSStateNotPresent = 0,
    /** GPS is locked. */
    PCRGPSStateLocked = 1,
    /**
     * GPS is locked and disciplining the system.
     * Disciplining must be explictely enabled to get to this state. */
    PCRGPSStateDisciplined = 2
} PCRGPSState;

/**
 * Controls the bit endianness of the Vita49 data stream when in Vita49 streaming mode.
 */
typedef enum PCREndianness {
    /** Big endian, default. */
    PCREndiannessBig = 0,
    /** Little endian. */
    PCREndiannessLittle = 1
} PCREndianness;

/**
 * Controls the detector used on sweep.
 */
typedef enum PCRSweepDetector {
    /** Average detector. */
    PCRSweepDetectorAvg = 0,
    /** Min hold detector. */
    PCRSweepDetectorMin = 1,
    /** Max hold detector. */
    PCRSweepDetectorMax = 2
} PCRSweepDetector;

/**
 * Controls the video processing (VBW) units on sweeps.
 */
typedef enum PCRSweepVideoUnits {
    /** dBm */
    PCRSweepVideoUnitsLog = 0,
    /** Linear voltage */
    PCRSweepVideoUnitsVoltage = 1,
    /** Linear power */
    PCRSweepVideoUnitsPower = 2,
    /** No VBW processing */
    PCRSweepVideoUnitsSample = 3
} PCRSweepVideoUnits;

/**
 * Controls the units of the returned sweep data.
 */
typedef enum PCRSweepScale {
    /** dBm */
    PCRSweepScaleLog = 0,
    /** mV */
    PCRSweepScaleLin = 1,
    /** dBFS, no corrections */
    PCRSweepScaleFullScale = 2
} PCRSweepScale;

/**
 * Controls the window used for sweeps.
 */
typedef enum PCRSweepWindow {
    /** SRS flattop */
    PCRSweepWindowFlattop = 0,
    /** Nutall window */
    PCRSweepWindowNutall = 1,
    /** Gaussian 6dB window used for EMC measurements and CISPR compatibility. */
    PCRSweepWindowGaussian6dB = 2
} PCRSweepWindow;

/**
 * Controls which trigger edge events to mark in the I/Q data stream.
 */
typedef enum PCRTriggerEdge {
    /** Rising edge triggering. */
    PCRTriggerEdgeRising = 0,
    /** Falling edge triggering. */
    PCRTriggerEdgeFalling = 1
} PCRTriggerEdge;

/**
 * Controls the I/Q stream mode. See @ref iqStreamingMode.
 */
typedef enum PCRStreamMode {
    PCRStreamModeLocal = 0,
    PCRStreamModeVRT = 1
} PCRStreamMode;

/**
 * Status code returned from all PCR API functions.
 */
typedef enum PCRStatus {
    /**
     * Returned when trying to finish a sweep that is not started.
     */
    PCRStatusSweepNotActiveErr = -14,
    /**
     * Returned when trying to start a sweep when one is already active.
     */
    PCRStatusSweepActiveErr = -13,
    /** 
     * Returned when the device detects issues on ADC clocks and sample alignment. 
     * Indicates possible hardware failure. Contact Signal Hound.
     */
    PCRStatusADCErr = -12,
    /** Returned when performing an invalid operation on a multi-receiver/paired system. */
    PCRStatusPairedErr = -11,
    /**
      * Returned when the device detects framing issue on measurement data
      * Measurement results are likely invalid. Device should be preset/power
      * cycled.
      */
    PCRStatusSyncErr = -10,
    /** Internal error. */
    PCRStatusIOErr = -9,
    /** One or more parameters provided were invalid. */
    PCRStatusInvalidParameterErr = -8,
    /**
     * Attempting to perform an operation that cannot currently be performed.
     * Often the result of trying to do something while the device is currently
     * making measurements or not in an idle state.
     */
    PCRStatusInvalidConfigurationErr = -7,
    /** Out of memory. */
    PCRStatusMemoryErr = -6,
    /**
     * No more devices can be interfaced. A device needs to be closed before
     * another can be connected.
     */
    PCRStatusMaxDevicesConnectedErr = -5,
    /** Device index specified is invalid. */
    PCRStatusInvalidDeviceErr = -4,
    /** One or more required pointer parameters are NULL. */
    PCRStatusNullPtrErr = -3,
    /** Device disconnected. Usually the result of network errors or UDP packet loss. */
    PCRStatusDeviceConnectionLostErr = -2,
    /** Unable to open device. */
    PCRStatusDeviceNotFoundErr = -1,

    /** Function returned successfully. */
    PCRStatusNoError = 0,

    /** One or more parameters were clamped to a valid range. */
    PCRStatusSettingClamped = 1,
    /** Measurement includes data subject to an ADC overload (clipping/compression) */
    PCRStatusADCOverflow = 2,
    /** Measurement uncalibrated. Overrides ADC overflow. Often the result of data loss. */
    PCRStatusUncalData = 3,
    /**
     * Temperature drift has occured since last configured.
     * Measurement error may be present.
     * Reinitialize the measurement to resolve.
     */
    PCRStatusTempDriftWarning = 4,
    /**
      * Indicates internal buffer wrap due to network throughput.
      * Data may be invalid
      */
    PCRStatusBufferOverflow = 5,
    /**
      * Calibration data potentially corrupt.
      */
    PCRStatusInvalidCalData = 6

} PCRStatus;

/** Struct populated in #pcrGetFullDeviceDiagnostics */
typedef struct PCRDeviceDiagnostics {
    /** Device voltage in V */
    float voltage;
    /** Input current in A */
    float currentInput;
    /** OCXO current in A */
    float currentOCXO;
    /** 5.8V power supply current in A */
    float current58;
    /** Internal FPGA temperature in C */
    float tempFPGA;
    /** Motherboard temp in C */
    float tempMB;
    /** Temperature on RF board 1 in C */
    float tempRFBoard1;
    /** Temperature on RF board 2 in C */
    float tempRFBoard2;
    /** SFP+ temperature in C */
    float sfpTemp;
    /** SFP+ Tx power in mW */
    float sfpTxPower;
    /** SFP+ Rx power in mW */
    float sfpRxPower;
} PCRDeviceDiagnostics;

#ifdef __cplusplus
extern "C" {
#endif

/**
 * Attempts to connect to a device. If the device is opened successfully, a
 * handle to the function will be returned through the device pointer. This
 * handle can then be used to refer to this device for all future API calls.
 * The device takes approximately 12 seconds to boot up after applying power.
 * Until the device is booted, this function will return device not found. The
 * PCR API does not set the SO_REUSEADDR socket option. For customers connecting
 * multiple networked devices, we recommend specifying the hostAddr explicitly
 * instead of using '0.0.0.0'. Especially for configurations that involve
 * multiple subnets. If not done, devices beyond the first will likely not be
 * found and this function will return an error.
 *
 * @param[out] handle Returns handle that can be used to interface the device.
 * Handle is only valid if this function returns successfully.
 *
 * @param[in] hostAddr Host interface IP on which the networked device is
 * connected, provided as a string. An example parameter is '192.168.2.2'.
 *
 * @param[in] deviceAddr Target device IP provided as a string. If more than
 * one device with this IP is connected to the host interface, the behavior is
 * undefined.
 *
 * @param[in] port Target device port.
 *
 * @return
 */
PCR_API PCRStatus pcrConnectDevice(int *handle,
                                   const char *hostAddr,
                                   const char *deviceAddr,
                                   uint16_t port);

/**
 * Pairs a secondary device to a primary device to create a multi-receiver system. 
 * The network parameters specified in this function are for the secondary device.
 * The secondary device should not be connected by the API before calling this funciton.
 * After this function returns successfully, the secondary device should not be
 * opened separately via the pcrConnectDevice function, or be paired again.
 * 
 * See @ref multiReceiver for more information.
 *
 * @param[in] handle Device on which the secondary unit will be paired.
 *
 * @param[in] hostAddr Host interface IP on which the networked device is
 * connected, provided as a string. An example parameter is '192.168.2.2'.
 *
 * @param[in] deviceAddr Target device IP provided as a string. If more than
 * one device with this IP is connected to the host interface, the behavior is
 * undefined.
 *
 * @param[in] port Target device port.
 * 
 * @return
 */
PCR_API PCRStatus pcrPairDevice(int handle, 
                                const char *hostAddr,
                                const char *deviceAddr, 
                                uint16_t port);

/**
 * Closes an open device. This function will release all resources for an open
 * device. When this function completes the handle parameter will no longer
 * point to a valid device. The device needs to be opened again to continue
 * using. All devices should be closed prior to the process exiting, but is not
 * strictly necessary.
 * 
 * If the handle refers to a primary device in a multi-receiver configuration, all
 * devices in that system are closed.
 *
 * @param[in] handle Device handle.
 *
 * @return
 */
PCR_API PCRStatus pcrCloseDevice(int handle);

/**
 * Retreive the device serial number.
 *
 * @param[in] handle Device handle.
 *
 * @param serialNumber Pointer to integer which will contain 8 digit serial
 * number if this function returns successfully.
 *
 * @return
 */
PCR_API PCRStatus pcrGetSerialNumber(int handle, int *serialNumber);

/**
 * Get the firmware version of the device.
 * The version is of the form 'major.minor.revision'.
 *
 * @param[in] handle Device handle.
 *
 * @param[out] major Version return as integer. Can be NULL.
 *
 * @param[out] minor Version return as integer.. Can be NULL.
 *
 * @param[out] revision Version return as integer. Can be NULL.
 *
 * @return
 */

PCR_API PCRStatus pcrGetFirmwareVersion(int handle, int *major, int *minor, int *revision);

/**
 * Returns the calibration date of the device. This is the date the unit was
 * factory adjusted. This value is primarily used for troubleshooting.
 *
 * @param[in] handle Device handle.
 *
 * @param lastCalDate Calibration date returned as seconds since epoch.
 *
 * @return
 */
PCR_API PCRStatus pcrGetCalDate(int handle, uint32_t *lastCalDate);

/**
 * Returns diagnostic information about a device.
 *
 * @param[in] handle Device handle.
 *
 * @param[out] voltage Voltage returned in Volts. Can be `NULL`.
 *
 * @param[out] current Current returned in Amps. Can be `NULL`.
 *
 * @param[out] temperature Temperature returned as Celcius. Can be `NULL`.
 *
 * @return
 */
PCR_API PCRStatus pcrGetDeviceDiagnostics(int handle, float *voltage, float *current, float *temperature);

/**
 * Returns extended diagnostic information.
 *
 * @param[in] handle Device handle.
 *
 * @param diagnostics Pointer to extended diagnostic struct.
 *
 * @return
 */
PCR_API PCRStatus pcrGetFullDeviceDiagnostics(int handle, PCRDeviceDiagnostics *diagnostics);

/**
 * Set the device into a lower power state. The device will typically consume 
 * ~30W in the low power state.
 * 
 * The device should be idle before calling this function. This function will fail silently 
 * otherwise.
 *
 * See @ref lowPowerMode for more information.
 * 
 * @param[in] handle Device handle.
 *
 * @return
 */
PCR_API PCRStatus pcrSetLowPowerState(int handle);

/**
 * Set the receiver to use either the internal time base reference or use a
 * 10MHz reference present on the 10MHz input port. The device must be in the
 * idle state (call #pcrAbort) for this function to take effect. If the
 * function returns successfully, verify the new state with the
 * #pcrGetReference function.
 *
 * @param[in] handle Device handle.
 *
 * @param[in] reference New reference state.
 *
 * @return
 */
PCR_API PCRStatus pcrSetReference(int handle, PCRReference reference);

/**
 * Get the current reference state.
 *
 * @param[in] handle Device handle.
 *
 * @param[out] reference Current reference configuration.
 * @return
 */
PCR_API PCRStatus pcrGetReference(int handle, PCRReference *reference);

/**
 * Set whether the GPS is allowed to discipline the timebased of the PCR.
 * Disabled by default. This function must be called in an idle state.
 * See @ref autoGPSTimebaseDiscipline for more information.
 *
 * @param[in] handle Device handle.
 *
 * @param[in] enabled True to enable GPS disciplining.
 *
 * @return
 */
PCR_API PCRStatus pcrSetGPSTimebaseUpdate(int handle, PCRBool enabled);

/**
 * Get the current auto GPS disciplining state.
 *
 * @param handle Device handle.
 *
 * @param enabled Returns true if GPS disciplining enabled.
 *
 * @return
 */
PCR_API PCRStatus pcrGetGPSTimebaseUpdate(int handle, PCRBool *enabled);

/**
 * Return information about the GPS holdover correction. Determine if a
 * correction exists and when it was generated.
 *
 * @param[in] handle Device handle.
 *
 * @param[out] usingGPSHoldover Returns whether the GPS holdover value is newer
 * than the factory calibration value. To determine whether the holdover value
 * is actively in use, you will need to use this function in combination with
 * pcrGetGPSState. This parameter can be NULL.
 *
 * @param[out] lastHoldoverTime If a GPS holdover value exists on the system,
 * return the timestamp of the value. Value is seconds since epoch. This
 * parameter can be NULL.
 *
 * @return
 */
PCR_API PCRStatus pcrGetGPSHoldoverInfo(int handle, PCRBool *usingGPSHoldover, uint64_t *lastHoldoverTime);

/**
 * @brief pcrGetGPSState Determine the GPS lock state of the PCR4200.
 *
 * @param[in] handle Device handle
 *
 * @param[out] state Pointer to instance of PCRGPSState.
 */
PCR_API PCRStatus pcrGetGPSState(int handle, PCRGPSState *state);

/**
 * Acquire the latest GPS information. The GPS info is updated once per second at
 * the PPS interval. This function can be called while measurements are active.
 *
 * @param[in] handle Device handle.
 *
 * @param[in] refresh When set to true and the device is not in a streaming
 * mode, the API will request the latest GPS information. Otherwise the last
 * retrieved data is returned.
 *
 * @param[out] updated Will be set to true if the NMEA data has been updated
 * since the last time the user called this function. Can be set to NULL.
 *
 * @param[out] secSinceEpoch Number of seconds since epoch as reported by the
 * GPS NMEA sentences. Last reported value by the GPS. If the GPS is not
 * locked, this value will be set to zero. Can be NULL.
 *
 * @param[out] latitude Latitude in decimal degrees. If the GPS is not locked,
 * this value will be set to zero. Can be NULL.
 *
 * @param[out] longitude Longitude in decimal degrees. If the GPS is not
 * locked, this value will be set to zero. Can be NULL.
 *
 * @param[out] altitude Altitude in meters. If the GPS is not locked, this
 * value will be set to zero. Can be NULL.
 *
 * @return
 */
PCR_API PCRStatus pcrGetGPSInfo(int handle, PCRBool refresh, PCRBool *updated, int64_t *secSinceEpoch,
    double *latitude, double *longitude, double *altitude);

/**
 * Set a custom key value pair for the U-blox F9 GPS Reciever. See @ref writingToGPS.
 *
 * @param[in] handle Device handle.
 *
 * @param[in] key Key used for configuring the F9 GPS Reciever
 *
 * @param[in] value Value used for configuring the F9 GPS Reciever
 *
 * @param[in] valueSizeBytes Size of the field being configured in bytes
 *
 * @return
 */
PCR_API PCRStatus pcrWriteMessageToGPS(int handle, uint32_t key, uint64_t value, int valueSizeBytes);

/**
 * Enable the fan. See @ref fanControl.
 *
 * @param[in] handle Device handle.
 *
 * @param[in] enabled True to enable fan.
 *
 * @return
 */
PCR_API PCRStatus pcrSetFanEnabled(int handle, PCRBool enabled);

/**
 * Return whether the fan is enabled.
 *
 * @param[in] handle Device handle.
 *
 * @param[out] enabled Returns true if enabled.
 *
 * @return
 */
PCR_API PCRStatus pcrGetFanEnabled(int handle, PCRBool *enabled);

/**
 * Configures the temperature/fan setpoint. See @ref fanControl. The
 * setpoint is clamped to #PCR_MIN_FAN_SETPOINT and #PCR_MAX_FAN_SETPOINT.
 *
 * @param[in] handle Device handle.
 *
 * @param setPoint New fan setpoint in C.
 *
 * @return
 */
PCR_API PCRStatus pcrSetFanSetPoint(int handle, float setPoint);

/**
 * Returns the configured fan setpoint.
 *
 * @param[in] handle Device handle.
 *
 * @param[out] setPoint Returns the current fan setpoint in C.
 *
 * @return
 */
PCR_API PCRStatus pcrGetFanSetPoint(int handle, float *setPoint);

/**
 * Helper function which returns the number of channels that can be configured.
 * This is equal to the number of devices in the system * 4.
 *
 * @param[in] handle Device handle.
 *
 * @param[out] channelCount Number of channels in system.
 *
 * @return
 */
PCR_API PCRStatus pcrGetChannelCount(int handle, int *channelCount);

/**
 * Configure the receiver channels. This function configures whether a channel
 * will be enabled for measurements, and whether one of those channels will be
 * used for sweeps.
 *
 * @param[in] handle Device handle.
 *
 * @param[in] enabled An array of integers equal in length to the number of
 * channels in the system. For each index in the array, a 0 or 1 indicates
 * whether that channel will be enabled in the measurements.
 *
 * For example, enabling channels (zero-based) 0,2,3 or (one-based) 1,3,4, the
 * array you provide to this parameter will look like [1,0,1,1]. If a channel
 * is designated as the sweep channel, it must also be enabled in this list.
 * Every channel enabled but not designated as the sweep channel will be an I/Q
 * streaming channel.
 *
 * @param[in] sweepChannel Specify a channel on which to perform sweeps. If no
 * channel is to perform sweeps, pass PCR_SWEEP_CHANNEL_DISABLED(-1) to this
 * parameter.
 *
 * @return
 */
PCR_API PCRStatus pcrSetChannelConfig(int handle, const int *enabled, int sweepChannel);

/**
 * Returns the current channel configuration.
 *
 * @param[in] handle Device handle.
 *
 * @param[out] enabled A pointer to an array of integers, equal in size to the number
 * of channels in the system. For each enabled channel, a 1 will be set in that
 * index, otherwise if that channel is disabled, a zero will be set. For
 * example, if channels (one-based) 1 & 2 are enabled, the array will be set to
 * {1,1,0,0}.
 *
 * @param[out] sweepChannel Returns which channel (if enabled) is set as the
 * sweeping channel. If no channel is set to sweep, -1 is returned.
 *
 * @return
 */
PCR_API PCRStatus pcrGetChannelConfig(int handle, int *enabled, int *sweepChannel);

/**
 * Set whether a specific channel should use the shared LO. This configuration
 * is independent of whether or not the channel is enabled. This setting only
 * affects channels enabled for I/Q streaming. A sweeping channel cannot use
 * the shared LO.
 *
 * @param[in] handle Device handle.
 *
 * @param[in] channel Channel between [0, channelCount-1].
 *
 * @param[in] useSharedLO True to use shared LO.
 *
 * @return
 */
PCR_API PCRStatus pcrSetChannelShared(int handle, int channel, PCRBool useSharedLO);

/**
 * This function is part of the fast retuning capabilities (See @ref fastRetuning).
 * 
 * Updates whether a specific channel should use the shared LO. This configuration
 * is independent of whether or not the channel is enabled. This setting only
 * affects channels enabled for I/Q streaming. A sweeping channel cannot use
 * the shared LO.
 * 
 * @param[in] handle Device handle.
 *
 * @param[in] channel Channel between [0, channelCount-1].
 *
 * @param[in] useSharedLO True to use shared LO.
 * 
 * @param[out] timeStamp If function returns successfully, this parameter returns a timestamp
 * in nanoseconds since epoch, representing the time at which the new channel configuration
 * has been applied. Can be NULL.
 *
 * @return Returns invalid configuration error if either the device is not actively streaming,
 * a channel is specified that is not streaming, or if a sweep is currently active.
 */
PCR_API PCRStatus pcrUpdateChannelShared(int handle, int channel, PCRBool useSharedLO, int64_t *timeStamp);

/**
 * Returns whether a channel is configured to use the shared LO.
 *
 * @param[in] handle Device handle.
 *
 * @param[in] channel Channel between [0, channelCount-1].
 *
 * @param[out] useSharedLO Returns true if the specified channel is configured to
 * use the shared LO.
 *
 * @return
 */
PCR_API PCRStatus pcrGetChannelShared(int handle, int channel, PCRBool *useSharedLO);

/**
 * Set a channel frequency. This frequency affects a channel that is configured
 * for I/Q streaming and to use the independent LO (not shared). This setting
 * persists independently of the other channel settings.
 *
 * @param[in] handle Device handle.
 *
 * @param[in] channel Channel between [0, channelCount-1].
 *
 * @param[in] freqHz Frequency of specified channel in Hz.
 *
 * @return
 */
PCR_API PCRStatus pcrSetChannelFreq(int handle, int channel, double freqHz);

/**
 * This function is part of the fast retuning capabilities (See @ref fastRetuning).
 *
 * Set a channel frequency. This frequency affects a channel that is configured
 * for I/Q streaming and to use the independent LO (not shared). This setting
 * persists independently of the other channel settings.
 *
 * @param[in] handle Device handle.
 *
 * @param[in] channel Channel between [0, channelCount-1].
 *
 * @param[in] freqHz Frequency of specified channel in Hz.
 * 
 * @param[out] timeStamp If function returns successfully, this parameter returns a timestamp
 * in nanoseconds since epoch, representing the time at which the new channel frequency
 * has been applied. Can be NULL.
 *
 * @return Returns invalid configuration error if either the device is not actively streaming,
 * a channel is specified that is not streaming, or if a sweep is currently active.
 */
PCR_API PCRStatus pcrUpdateChannelFreq(int handle, int channel, double freqHz, int64_t *timeStamp);

/**
 *Returns a channels configured frequency. This is the frequency of a channel
 *that is configured for I/Q streaming and to use the independent LO.
 *
 * @param[in] handle Device handle.
 *
 * @param[in] channel Channel between [0, channelCount-1].
 *
 * @param[out] freqHz Returns the frequency of the specified channel in Hz.
 *
 * @return
 */
PCR_API PCRStatus pcrGetChannelFreq(int handle, int channel, double *freqHz);

/**
 * Sets a fixed phase offset for a single channel. This offset is applied 
 * as a complex phase rotation on each sample. This offset can be used to
 * provide phase alignment between multiple channels. The default offset
 * for each channel is 0 radians.
 *
 * See @ref phaseCoherence for more information.
 * 
 * @param[in] handle Device handle.
 *
 * @param[in] channel Channel between [0, channelCount-1].
 *
 * @param[in] phaseOffset Sets a fixed phase offset to apply to the specified
 * channel. This offset is only applied when streaming I/Q data. This offset is
 * specified in radians.
 *
 * @return
 */
PCR_API PCRStatus pcrSetChannelPhaseOffset(int handle, int channel, double phaseOffset);

/**
 * Get the phase offset current applied to a channel.
 *
 * See @ref phaseCoherence for more information.
 * 
 * @param[in] handle Device handle.
 *
 * @param[in] channel Channel between [0, channelCount-1].
 *
 * @param[out] phaseOffset Returns the phase offset set for the specified
 * channel as radians.
 *
 * @return
 */
PCR_API PCRStatus pcrGetChannelPhaseOffset(int handle, int channel, double *phaseOffset);

/**
 * Set a channels timing delay. This delay is specified in seconds, and is 
 * applied as part of an interpolating FIR filter. This delay
 * can be used as part of a system requiring phase and time alignment across the
 * 40MHz bandwidth between multiple channels.
 *
 * See @ref phaseCoherence and @ref phaseCalibration for more information.
 *
 * @param[in] handle Device handle.
 *
 * @param[in] channel Channel between [0, channelCount-1].
 *
 * @param[in] delay Channel delay to apply in seconds. Delay should be in 
 * the range [-25e-9, +25e-9] or [-25ns, +25ns]. 
 *
 * @return
 */
PCR_API PCRStatus pcrSetChannelDelay(int handle, int channel, double delay);

/**
 * Retrieve a configured channels timing delay.
 *
 * See @ref phaseCoherence for more information.
 * 
 * @param[in] handle Device handle.
 *
 * @param[in] channel Channel between [0, channelCount-1].
 *
 * @param[out] delay Returns the applied delay in seconds.
 *
 * @return
 */
PCR_API PCRStatus pcrGetChannelDelay(int handle, int channel, double *delay);

/**
 * Sets the frequency of the measurement to be used for all streaming I/Q
 * channels that are using the shared LO. This frequency is also shared with the
 * VSG.
 *
 * @param[in] handle Device handle.
 *
 * @param[in] freqHz Frequency specified in Hz.
 *
 * @return
 */
PCR_API PCRStatus pcrSetSharedFreq(int handle, double freqHz);

/**
 * This function is part of the fast retuning capabilities (See @ref fastRetuning).
 *
 * Updates the frequency of the measurement to be used for all streaming I/Q
 * channels that are using the shared LO. This frequency is also shared with the
 * VSG.
 * 
 * @param[in] handle Device handle.
 *
 * @param[in] freqHz Frequency specified in Hz.
 * 
 * @param[out] timeStamp Timestamp in ns since epoch. Marks the time at which the new
 * shared frequency has been set. If no channel is currently using the shared LO, the
 * timestamp will not be valid and should be ignored. Can be NULL.
 *
 * @return Returns invalid configuration error if either the device is not actively streaming,
 * or if a sweep is not currently active.
 */
PCR_API PCRStatus pcrUpdateSharedFreq(int handle, double freqHz, int64_t *timeStamp);

/**
 * Retrieve the frequency of channels using the shared LO.
 *
 * @param[in] handle Device handle.
 *
 * @param[out] freqHz Returns the frequency specified in Hz.
 *
 * @return
 */
PCR_API PCRStatus pcrGetSharedFreq(int handle, double *freqHz);

/**
 * Sets the I/Q stream data type. See @ref iqDataTypes for more information.
 *
 * @param[in] handle Device handle.
 *
 * @param[in] dataType Data type.
 *
 * @return
 */
PCR_API PCRStatus pcrSetStreamDataType(int handle, PCRDataType dataType);

/**
 * Sets whether the data returned is in the units of dBm or dBFS. This only
 * applies for local streaming mode and 32-bit data types. 16-bit data types
 * are always returned as full scale. Full scale is disabled by default.
 *
 * @param[in] handle Device handle.
 *
 * @param[in] fullScale True if data should be returned as full scale.
 *
 * @return
 */
PCR_API PCRStatus pcrSetStreamFullScale(int handle, PCRBool fullScale);

/**
 * Set the reference level of the I/Q data streams. This reference level is used
 * for all streaming channels. See @ref refLevelAndSensitivity for more information.
 * If attenuation is not set to auto, this value is ignored.
 *
 * @param[in] handle Device handle.
 *
 * @param[in] refLevel Reference level specified as dBm.
 *
 * @return
 */
PCR_API PCRStatus pcrSetStreamRefLevel(int handle, double refLevel);

/**
 * Set the attenuation for the I/Q data streams. This applies to all streaming
 * channels. See @ref refLevelAndSensitivity for more information. Attenuation
 * is set to auto by default.
 *
 * @param[in] handle Device handle.
 *
 * @param[in] atten Attenuation between [0,6], or -1 for auto (use reference level).
 *
 * @return
 */
PCR_API PCRStatus pcrSetStreamAtten(int handle, int atten);

/**
 * Set the sample rate for the I/Q data streams. This applies to all streaming
 * channels. See @ref iqSampleRates for more information.
 *
 * @param[in] handle Device handle.
 *
 * @param[in] decimation Sets the sample rate to 50MS/s / decimation.
 * Decimation must be a power of 2, and must be between 1 and 256.
 *
 * @return
 */
PCR_API PCRStatus pcrSetStreamSampleRate(int handle, int decimation);

/**
 * Set the streaming bandwidth. This setting applies to all streaming channels.
 * See @ref iqSampleRates for more information.
 *
 * @param[in] handle Device handle.
 *
 * @param[in] bandwidth Bandwidth specified in Hz.
 *
 * @return
 */
PCR_API PCRStatus pcrSetStreamBandwidth(int handle, double bandwidth);

/**
 * Set the streaming external trigger edge. External triggers will appear only
 * for the specified edges.
 *
 * @param[in] handle Device handle.
 *
 * @param[in] edge Trigger edge.
 *
 * @return
 */
PCR_API PCRStatus pcrSetStreamTriggerEdge(int handle, PCRTriggerEdge edge);

/**
 * Set the I/Q stream mode. See @ref iqStreamingMode for more information.
 *
 * @param[in] handle Device handle.
 *
 * @param[in] mode Mode.
 *
 * @return
 */
PCR_API PCRStatus pcrSetStreamMode(int handle, PCRStreamMode mode);

/**
 * Set the I/Q streaming port used for local streaming. By default, it's set to 
 * 4991. For customers using the local stream mode, this function should
 * usually not need to be called, unless the default port conflicts with
 * another application.
 * 
 * One reason this might need to be called is for multi-receiver systems.
 * If all devices in a multi-receiver system are on the same subnet, the default
 * port number between multiples devices can conflict. For that reason changing each
 * devices port number will be needed.
 * 
 * This function sets one devices port at a time.
 * 
 * See @ref iqStreamingMode for more information.
 *
 * @param[in] handle Device handle.
 * 
 * @param[in] deviceIndex Index into multi receiver system. 0 is the primary device,
 * 1,2,3 are the secondary devices (if paired). For a single receiver (4 channel system),
 * this value should be zero.
 *
 * @param[in] port Network port to set.
 *
 * @return
 */
PCR_API PCRStatus pcrSetStreamLocalAddr(
    int handle,
    int deviceIndex,
    uint16_t port);

/**
 * Set the destination network addess for the I/Q data in the Vita49
 * streaming mode. 
 *
* See @ref iqStreamingMode for more information.
 *
 * @param[in] handle Device handle.
 * 
 * @param[in] channel Channel between [0, channelCount-1].
 *
 * @param[in] dstAddr IP address specified as a string, for example "192.168.3.10".
 *
 * @param[in] dstPort Network port.
 *
 * @param[in] dstMACAddr MAC address.
 *
 * @return
 */
PCR_API PCRStatus pcrSetStreamVrtAddr(
    int handle,
    int channel, 
    const char *dstAddr, 
    uint16_t dstPort, 
    uint64_t dstMACAddr);

/**
 * Set the stream ID of the Vita49 stream. This is the ID returned in the
 * streams context packet. Each channel will have it's own stream ID.
 *
 * @param[in] handle Device handle.
 *
 * @param[in] channel Channel between [0, channelCount-1].
 *
 * @param[in] streamID StreamID.
 *
 * @return
 */
PCR_API PCRStatus pcrSetStreamVrtChannelID(int handle, int channel, int streamID);

/**
 * Sets the interval for the context packets sent in Vita49 stream mode. All
 * channels will adhere to the context interval.
 *
 * @param[in] handle Device handle.
 *
 * @param[in] enabled Enables context packets. Enabled by default.
 *
 * @param[in] interval Sets the interval. 64 by default.
 *
 * @return
 */
PCR_API PCRStatus pcrSetStreamVrtCntxInterval(int handle, PCRBool enabled, int interval);

/**
 * Set the endianness of the Vita49 data. Vita49 data is big endian by default.
 * Many modern PCs are little endian. A PC that is receiving and processing the
 * Vita49 data may want to eliminate the need to change the byte ordering.
 *
 * @param[in] handle Device handle.
 *
 * @param[in] endianness The endianness to use.
 *
 * @return
 */
PCR_API PCRStatus pcrSetStreamVrtEndianness(int handle, PCREndianness endianness);

/**
 * The reference level controls the sensitivity of the receiver by setting the
 * attenuation to optimize measurements for signals at or below the reference
 * level. See @ref refLevelAndSensitivity for more information. Attenuation
 * must be set to automatic (-1) to set reference level.
 *
 * This function sets the reference level for sweeps only.
 *
 * @param[in] handle Device handle.
 *
 * @param[in] refLevel Set the reference level of the receiver in dBm.
 *
 * @return
 */
PCR_API PCRStatus pcrSetSweepRefLevel(int handle, double refLevel);

/**
 * Set the device attenuation. See @ref refLevelAndSensitivity for more
 * information. Valid values for attenuation are between [0,6] representing
 * between [0,30] dB of attenuation (5dB steps). Setting the attenuation to -1
 * tells the receiver to automatically choose the best attenuation value for
 * the specified reference level selected. Setting attenuation to a non-auto
 * value overrides the reference level selection. The header file provides the
 * PCR_AUTO_ATTEN macro for -1.
 *
 * This function sets the attenuation for sweeps only.
 *
 * @param[in] handle Device handle.
 *
 * @param[in] atten Attenuation value between [-1,6].
 *
 * @return
 */
PCR_API PCRStatus pcrSetSweepAtten(int handle, int atten);

/**
 * Set the sweep center and span frequencies. The new configuration cannot
 * cause the sweep to exceed the frequency range of the instrument. If that
 * occurs, the settings will be clamped to the range of the instrument.
 *
 * @param[in] handle Device handle.
 *
 * @param[in] centerHz Center frequency in Hz.
 *
 * @param[in] spanHz Span in Hz.
 *
 * @return
 */
PCR_API PCRStatus pcrSetSweepCenterSpan(int handle, double centerHz, double spanHz);

/**
 * Set the sweep start and stop frequency. The new configuration cannot
 * cause the sweep to exceed the frequency range of the instrument. If that
 * occurs, the settings will be clamped to the range of the instrument.
 *
 * @param[in] handle Device handle.
 *
 * @param[in] startHz Start frequency in Hz.
 *
 * @param[in] stopHz Stop frequency in Hz.
 *
 * @return
 */
PCR_API PCRStatus pcrSetSweepStartStop(int handle, double startHz, double stopHz);

/**
 * Set sweep resolution and video bandwidth.
 *
 * @param[in] handle Device handle.
 *
 * @param[in] rbw Resolution bandwidth in Hz.
 *
 * @param[in] vbw Video bandwidth in Hz. Cannot be greater than RBW.
 *
 * @param[in] sweepTime Suggest the total acquisition time of the sweep.
 * Specified in seconds. This parameter is a suggestion and will ensure RBW and
 * VBW are first met before increasing sweep time.
 *
 * @return
 */
PCR_API PCRStatus pcrSetSweepCoupling(int handle, double rbw, double vbw, double sweepTime);

/**
 * Set sweep detector.
 *
 * @param[in] handle Device handle.
 *
 * @param[in] detector Sweep detector.
 *
 * @param[in] videoUnits Video processing units.
 *
 * @return
 */
PCR_API PCRStatus pcrSetSweepDetector(int handle, PCRSweepDetector detector, PCRSweepVideoUnits videoUnits);

/**
 * Set the sweep mode output unit type.
 *
 * @param[in] handle Device handle.
 *
 * @param[in] scale Sweep units.
 *
 * @return
 */
PCR_API PCRStatus pcrSetSweepScale(int handle, PCRSweepScale scale);

/**
 * Set sweep mode window function.
 *
 * @param[in] handle Device handle.
 *
 * @param[in] window Window function.
 *
 * @return
 */
PCR_API PCRStatus pcrSetSweepWindow(int handle, PCRSweepWindow window);

/**
 * Enable/disable the VSG output. When the VSG is enabled, it will output
 * at the configured shared LO frequency.
 * 
 * See @ref vsg for more information.
 *
 * @param[in] handle Device handle.
 *
 * @param[in] enabled True to enable.
 *
 * @return
 */
PCR_API PCRStatus pcrSetVSGEnabled(int handle, PCRBool enabled);

/**
 * Set the VSG output level. The valid range of output levels is [10,-80].
 * Not every frequency is capable of outputting +10dBm, and as such the output
 * power might be lower than expected. See the product manual for a typical
 * maximum output power curve across frequency. 
 * 
 * See @ref vsg for more information.
 *
 * @param[in] handle Device handle.
 *
 * @param[in] level Output level in dBm.
 *
 * @return
 */
PCR_API PCRStatus pcrSetVSGLevel(int handle, double level);

/**
 * Set the VSG output pattern. 
 * 
 * See @ref vsg for more information
 *
 * @param[in] handle Device handle.
 *
 * @param[in] iq The pattern supplied as complex values in the data type
 * specified by dataType. The pattern should be a basedband I/Q waveform specified
 * at 125MS/s with at most 40MHz of bandwidth. 
 *
 * @param[in] len The length of the pattern in complex samples. The pattern must
 * be a multiple of 16, and must be between [16,4096] samples.
 *
 * @param[in] dataType Specifies the data type of the provided pattern. Choices
 * are 16-bit interleaved complex integers and 32-bit complex interleaved
 * floating point values.
 *
 * @return
 */
PCR_API PCRStatus pcrSetVSGPattern(int handle, void *iq, int len, PCRDataType dataType);

/**
 * Start the measurement. At this point, all configuration routines must be
 * completed. Any change to the measurement configuration after calling this
 * function will not affect the measurement until the measurement is restarted.
 *
 * This function can be called while the measurements are active, and it will
 * have the affect of calling #pcrAbort and #pcrInitiate.
 *
 * @param[in] handle Device handle.
 *
 * @return
 */
PCR_API PCRStatus pcrInitiate(int handle);

/**
 * Stops the active measurement. All function calls to request data must be
 * completed before calling this function. When this function returns, the
 * device will be in an idle state.
 *
 * @param[in] handle Device handle.
 *
 * @return
 */
PCR_API PCRStatus pcrAbort(int handle);

/**
 * Retrieve the I/Q measurement parameters for an active I/Q stream. This
 * function can only be called if the measurement is active.
 *
 * @param[in] handle Device handle.
 *
 * @param[out] channelCount Returns the number of streaming channels. Can be NULL.
 *
 * @param[out] sampleRate The sample rate in Hz. Can be NULL.
 *
 * @param[out] bandwidth The bandwidth of the I/Q capture in Hz. Can be NULL.
 *
 * @return
 */
PCR_API PCRStatus pcrStreamParameters(int handle, int *channelCount, double *sampleRate, double *bandwidth);

/**
 * Retrieve the I/Q correction factor for an active I/Q stream. Each channel
 * will have it's own correction factor. See @ref iqDataTypes for more
 * information.
 *
 * @param[in] handle Device handle.
 *
 * @param[in] channel Channel between [0, channelCount-1].
 *
 * @param[out] correction Amplitude correction used by the API to convert from full
 * scale I/Q to amplitude corrected I/Q.
 *
 * @return
 */
PCR_API PCRStatus pcrStreamCorrection(int handle, int channel, float *correction);

/**
 * Retrieve one block of I/Q data as specified by the user. This function
 * blocks until the data requested is available.
 *
 * This function should only be used when in local I/Q stream mode.
 *
 * @param[in] handle Device handle.
 *
 * @param[out] buffers Pointer to user allocated buffers to recieve data.
 * buffers should point to an array of pointers, and that array should be equal
 * to the number of active I/Q channels. Each pointer in that buffer should
 * point to an array large enough to received the I/Q data specified by both the
 * iqBufSize and iqDataType parameter.
 *
 * @param[in] samples Specifies the number of I/Q samples, per channel, to be
 * retrieved. Must be greater than zero.
 *
 * @param[out] nsSinceEpoch Nanoseconds since epoch. The time of the first I/Q
 * sample returned. Can be NULL. See @ref IQTimestamping for more information.
 *
 * @param[in] purge When set to true, any buffered I/Q data in the API is
 * purged before returned beginning the I/Q block acquisition.
 *
 * @param[out] sampleLoss Set by the API when a sample loss condition occurs.
 * If enough I/Q data has accumulated in the internal API circular buffer, the
 * buffer is cleared and the sample loss flag is set. If purge is set to true,
 * the sample flag will always be set to false. Can be NULL.
 *
 * @param[out] samplesRemaining Set by the API, returns the number of samples
 * remaining in the internal I/Q circular buffer. Can be NULL.
 *
 * @param[out] triggers Pointer to user allocated array of trigger indices. The
 * buffer must be at least triggerBufSize contiguous doubles. The pointer can
 * also be NULL to indicate you do not wish to receive external trigger
 * information. See @ref iqStreaming section for more information on triggers.
 *
 * @param[in] triggerBufSize Specifies the size of the triggersr array. If the
 * triggers array is NULL, this value should be zero.
 *
 * @return
 */
PCR_API PCRStatus pcrStreamRecv(int handle,
                                void **buffers,
                                int samples,
                                int64_t *nsSinceEpoch,
                                PCRBool purge,
                                int *sampleLoss,
                                int *samplesRemaining,
                                double *triggers,
                                int triggerBufSize);

/**
 * Retrieve one block of I/Q data as specified by the user. 
 * This function flattens the data and returns it as a 1d array.
 * This function blocks until the data requested is available.
 *
 * This function should only be used when in local I/Q stream mode.
 *
 * @param[in] handle Device handle.
 *
 * @param[out] buffer Pointer to user allocated buffers to recieve data.
 * The array should be large enough to received the I/Q data specified by 
 * iqBufSize, iqDataType, and the number of active streaming channels.
 *
 * @param[in] samples Specifies the number of I/Q samples, per channel, to be
 * retrieved. Must be greater than zero.
 *
 * @param[out] nsSinceEpoch Nanoseconds since epoch. The time of the first I/Q
 * sample returned. Can be NULL. See @ref IQTimestamping for more information.
 *
 * @param[in] purge When set to true, any buffered I/Q data in the API is
 * purged before returned beginning the I/Q block acquisition.
 *
 * @param[out] sampleLoss Set by the API when a sample loss condition occurs.
 * If enough I/Q data has accumulated in the internal API circular buffer, the
 * buffer is cleared and the sample loss flag is set. If purge is set to true,
 * the sample flag will always be set to false. Can be NULL.
 *
 * @param[out] samplesRemaining Set by the API, returns the number of samples
 * remaining in the internal I/Q circular buffer. Can be NULL.
 *
 * @param[out] triggers Pointer to user allocated array of trigger indices. The
 * buffer must be at least triggerBufSize contiguous doubles. The pointer can
 * also be NULL to indicate you do not wish to receive external trigger
 * information. See @ref iqStreaming section for more information on triggers.
 *
 * @param[in] triggerBufSize Specifies the size of the triggersr array. If the
 * triggers array is NULL, this value should be zero.
 *
 * @return
 */
PCR_API PCRStatus pcrStreamRecvFlat(int handle,
                                void* buffer,
                                int samples,
                                int64_t* nsSinceEpoch,
                                PCRBool purge,
                                int* sampleLoss,
                                int* samplesRemaining,
                                double* triggers,
                                int triggerBufSize);

/**
 * Retrieves the sweep parameters for an active sweep measurement. This function
 * can only be called if the measurement is active.
 *
 * @param[in] handle Device handle.
 *
 * @param[out] actualRBW Returns the RBW being used in Hz. Can be NULL.
 *
 * @param[out] actualVBW Returns the VBW being used in Hz. Can be NULL.
 *
 * @param[out] actualStartFreq Returns the frequency of the first
 * bin in Hz. Can be NULL.
 *
 * @param[out] binSize Returns the frequency spacing between each frequency bin
 * in the sweep in Hz.
 *
 * @param[out] sweepSize Returns the length of the sweep (number of frequency
 * bins). Can be NULL.
 *
 * @return
 */
PCR_API PCRStatus pcrSweepGetParameters(int handle,
                                        double *actualRBW,
                                        double *actualVBW,
                                        double *actualStartFreq,
                                        double *binSize,
                                        int *sweepSize);

/**
 * Perform a single sweep. Block until the sweep completes. Internally, this
 * function is implemented as calling #pcrSweepStart followed by
 * #pcrSweepFinish. Do not mix calls to this function and the asynchronous
 * versions.
 *
 * @param[in] handle Device handle.
 *
 * @param[out] sweep Pointer to array equal in length to the sweep size.
 *
 * @return
 */
PCR_API PCRStatus pcrGetSweep(int handle, float *sweep);

/**
 * Start a sweep. If successful, this function returns immediately. See #pcrSweepFinish.
 *
 * @param[in] handle Device handle.
 *
 * @return
 */
PCR_API PCRStatus pcrSweepStart(int handle);

/**
 * Finish a previously started queued sweep. Blocks until the sweep completes. 
 * The sweep must be finished by the same thread that started it.
 *
 * @param[in] handle Device handle.
 *
 * @param[out] sweep Pointer to array equal in length to the sweep size.
 *
 * @return
 */
PCR_API PCRStatus pcrSweepFinish(int handle, float *sweep);

/**
 * This function is part of a group of functions used to configure the network
 * settings of the device over the USB 2.0 port. The handle used for these
 * functions can only be used with the other network config functions.
 *
 * @param[out] serials Array of ints. Must be as big as deviceCount. Returns a
 * list of all PCR devices connected to the PC over USB.
 *
 * @param[inout] deviceCount Point to int that contains the size of the serials
 * array. When the function returns it will contain the number of devices
 * returned.
 *
 * @return
 */
PCR_API PCRStatus pcrNetworkConfigGetDeviceList(int *serials, int *deviceCount);

/**
 * This function is part of a group of functions used to configure the network
 * settings of the device over the USB 2.0 port. The handle used for these
 * functions can only be used with the other network config functions.
 *
 * @param[out] handle If successful, handle will return an integer handle that
 * can be used to configure the network configuration through the
 * pcrNetworkConfig*** functions.
 *
 * @param[in] serialNumber Serial number of the device to open. A list of
 * connected serial numbers can be retrieved from the
 * #pcrNetworkConfigGetDeviceList function.
 *
 * @return
 */
PCR_API PCRStatus pcrNetworkConfigOpenDevice(int *handle, int serialNumber);

/**
 * This function is part of a group of functions used to configure the network
 * settings of the device over the USB 2.0 port. The handle used for these
 * functions can only be used with the other network config functions.
 *
 * Closes the device and frees any resources. The handle should be closed
 * before interfacing the device through the main API interface.
 *
 * @param[in] handle Device handle returned from #pcrNetworkConfigOpenDevice.
 *
 * @return
 */
PCR_API PCRStatus pcrNetworkConfigCloseDevice(int handle);

/**
 * This function is part of a group of functions used to configure the network
 * settings of the device over the USB 2.0 port. The handle used for these
 * functions can only be used with the other network config functions.
 *
 * Retrieve the device MAC address.
 *
 * @param[in] handle Device handle returned from #pcrNetworkConfigOpenDevice.
 *
 * @param[out] mac Pointer to char buffer, to contain the null terminated MAC
 * address string of the unit, with the following format “XX-XX-XX-XX-XX-XX”.
 * Must be large enough to accommodate this string including null termination.
 *
 * @return
 */
PCR_API PCRStatus pcrNetworkConfigGetMAC(int handle, char *mac);

/**
 * This function is part of a group of functions used to configure the network
 * settings of the device over the USB 2.0 port. The handle used for these
 * functions can only be used with the other network config functions.
 *
 * Set device IP address.
 *
 * @param[in] handle Device handle returned from #pcrNetworkConfigOpenDevice.
 *
 * @param[in] addr pointer to char buffer, to contain the null terminated IP
 * address string of the form “xxx.xxx.xxx.xxx”. For functions retrieving the
 * IP address, the buffer must be large enough to hold this string including
 * null termination.
 *
 * @param[in] nonVolatile When set to true, setting applied will be written
 * to internal flash, which will persist through a device power cycle.
 *
 * @return
 */
PCR_API PCRStatus pcrNetworkConfigSetIP(int handle, const char *addr, PCRBool nonVolatile);

/**
 * This function is part of a group of functions used to configure the network
 * settings of the device over the USB 2.0 port. The handle used for these
 * functions can only be used with the other network config functions.
 *
 * Get the device IP address.
 *
 * @param[in] handle Device handle returned from #pcrNetworkConfigOpenDevice.
 *
 * @param[out] addr pointer to char buffer, to contain the null terminated IP
 * address string of the form “xxx.xxx.xxx.xxx”. For functions retrieving the
 * IP address, the buffer must be large enough to hold this string including
 * null termination.
 *
 * @return
 */
PCR_API PCRStatus pcrNetworkConfigGetIP(int handle, char *addr);

/**
 * This function is part of a group of functions used to configure the network
 * settings of the device over the USB 2.0 port. The handle used for these
 * functions can only be used with the other network config functions.
 *
 * Set the device network port number.
 *
 * @param[in] handle Device handle returned from #pcrNetworkConfigOpenDevice.
 *
 * @param[out] port Port number.
 *
 * @param[in] nonVolatile When set to true, setting applied will be written
 * to internal flash, which will persist through a device power cycle.
 *
 * @return
 */
PCR_API PCRStatus pcrNetworkConfigSetPort(int handle, int port, PCRBool nonVolatile);

/**
 * This function is part of a group of functions used to configure the network
 * settings of the device over the USB 2.0 port. The handle used for these
 * functions can only be used with the other network config functions.
 *
 * Get the device network port number.
 *
 * @param[in] handle Device handle returned from #pcrNetworkConfigOpenDevice.
 *
 * @param[out] port Port number.
 *
 * @return
 */
PCR_API PCRStatus pcrNetworkConfigGetPort(int handle, int *port);

/**
 * This function is used as part of the configuration for I/Q timestamping with an
 * external GPS. See the automatic approach referenced here @ref IQTimestampingExternal.
 *
 * @param[in] handle Device handle.
 *
 * @param[in] useExternalPPS Set to true when configuring the device for I/Q timestamping
 * with an external GPS.
 *
 * @return
 */
PCR_API PCRStatus pcrSetPPSSource(int handle, PCRBool useExternalPPS);

/**
 * This function is used as part of the configuration for I/Q timestamping with an
 * external GPS. See the automatic approach referenced here @ref IQTimestampingExternal.
 * 
 * This function can only be called if PPS source is set to external, I/Q streaming is
 * active, and no sweeps are currently being performed.
 *
 * @param[in] handle Device handle.
 *
 * @param[in] epochTimeSec The current time in seconds since epoch.
 *
 * @return
 */
PCR_API PCRStatus pcrSetTime(int handle, uint32_t epochTimeSec);


/**
 * Get the API version.
 *
 * @return
 * The returned string is of the form
 *
 * major.minor.revision
 *
 * Ascii periods ('.') separate positive integers. Major/minor/revision are not
 * guaranteed to be a single decimal digit. The string is null terminated. The
 * string should not be modified or freed by the user. An example string is
 * below…
 *
 * ['3' | '.' | '0' | '.' | '1' | '1' | '\0'] = "3.0.11"
 */
PCR_API const char* pcrGetAPIVersion();

/**
 * Retrieve a descriptive string of a SmStatus enumeration. Useful for
 * debugging and diagnostic purposes.
 *
 * @param[in] status Status code returned from any API function.
 *
 * @return
 */
PCR_API const char* pcrGetErrorString(PCRStatus status);

#ifdef __cplusplus
} // Extern "C"
#endif

#endif // PCR_API_H
