import math
import time
from datetime import datetime
from datetime import timedelta
from datetime import timezone

class Timestamp:
    """
    Represents a high-precision timestamp

    If we use a single float64 to represent a timestamp in seconds since the Unix epoch (Jan. 1, 1970), we will only
    maintain precision to several microseconds.  This is not precise enough for many applications, so this type stores
    timestamps using two float64 values: one for whole seconds, one for fractional seconds.  After all operations on
    the timestamp, this whole/fractional split is maintained.
    """
    whole: float
    """Whole second part of the timestamp"""
    fractional: float
    """Fractional second part of the timestamp"""

    def __init__(self, whole:float = 0.0, fractional:float = 0.0):
        """Constructs a timestamp"""
        self.whole = whole
        self.fractional = fractional
        self._normalize()

    def __add__(self, other) -> 'Timestamp':
        """Adds two timestamps"""
        return Timestamp(self.whole + other.whole, self.fractional + other.fractional)

    def __iadd__(self, other) -> 'Timestamp':
        """Adds two timestamps in-place"""
        self.whole += other.whole
        self.fractional += other.fractional
        self._normalize()
        return self

    def __sub__(self, other) -> 'Timestamp':
        """Subtracts two timestamps"""
        return Timestamp(self.whole - other.whole, self.fractional - other.fractional)

    def __isub__(self, other) -> 'Timestamp':
        """Subtracts two timestamps in-place"""
        self.whole -= other.whole
        self.fractional -= other.fractional
        self._normalize()
        return self

    def __eq__(self, other):
        """Equality comparison between two timestamps"""
        return (self.whole, self.fractional) == (other.whole, other.fractional)

    def __str__(self) -> str:
        """Returns string representation of the timestamp"""
        return self.utc_datetime().__str__()

    def utc_datetime(self) -> datetime:
        """Returns a UTC datetime object for the timestamp"""
        dt = datetime.fromtimestamp(self.whole, timezone.utc)
        dt += timedelta(microseconds=self.fractional * 1e6)
        return dt

    def local_datetime(self) -> datetime:
        """Returns a local timezone datetime object for the timestamp"""
        dt = datetime.fromtimestamp(self.whole)
        dt += timedelta(microseconds=self.fractional * 1e6)
        return dt

    def iso8601(self) -> str:
        """Returns an ISO 8601 string representation of the timestamp"""
        dt = datetime.fromtimestamp(self.whole, timezone.utc)
        dt += timedelta(microseconds=self.fractional * 1e6)
        return dt.isoformat()

    def _normalize(self):
        """Normalizes the split between whole and fractional parts of the timestamp"""
        f, w = math.modf(self.whole)
        self.fractional += f
        self.whole = w

        f, w = math.modf(self.fractional)
        self.whole += w
        self.fractional = f

        if (self.whole < 0.0) and (self.fractional > 0.0):
            self.fractional -= 1.0
            self.whole += 1.0
        elif (self.whole > 0.0) and (self.fractional < 0.0):
            self.fractional += 1.0
            self.whole -= 1.0

    @staticmethod
    def now() -> 'Timestamp':
        """Returns the current system time timestamp"""
        t = time.time_ns()
        whole = math.floor(t / 1e9)
        frac = (t - (whole * 1e9)) / 1e9
        return Timestamp(whole, frac)
