# This example configures the onboard signal generator for a CW output
# and configures a single channel I/Q reciever.

from pcrdevice.pcr_api import *

from math import log10

def avg_power_dbm(src: np.typing.NDArray[np.complex64], length: int) -> float:
    p = 0.0
    for i in range(length):
        p += src[i].real * src[i].real + src[i].imag * src[i].imag

    return 10.0 * log10(p / length)

def pcr_example_vsg_and_rx():
    # Decimation should be a power of 2
    DECIMATION = 16
    EXPECTED_SAMPLE_RATE = PCR_STREAM_SR / DECIMATION
    frequency = 2.0e9

    # Connect the device
    status, handle = pcr_connect_device(PCR_DEFAULT_HOST_ADDR, PCR_DEFAULT_DEVICE_ADDR, PCR_DEFAULT_PORT).values()
    if status != PCRStatus.PCRStatusNoError:
        print(f'Error opening device: {pcr_get_error_string(status)}')
        return

    # Enable one stream channel, between [0,3]
    ACTIVE_CHANNEL = 0
    ch_enabled = [False, False, False, False]
    ch_enabled[ACTIVE_CHANNEL] = True

    # Set channel configuration
    sts = pcr_set_channel_config(handle, ch_enabled, PCR_SWEEP_CHANNEL_DISABLED)
    sts = pcr_set_channel_shared(handle, ACTIVE_CHANNEL, True)
    # VSG uses the shared frequency. So this function will not only control
    #  any receive channels using the shared frequency, but also the VSG output frequency.
    # For this example, the recieve channel will use the shared frequency.
    sts = pcr_set_shared_freq(handle, frequency)

    # Configure I/Q stream
    sts = pcr_set_stream_data_type(handle, PCRDataType.PCRDataType32fc)
    sts = pcr_set_stream_ref_level(handle, -20.0)
    sts = pcr_set_stream_atten(handle, PCR_AUTO_ATTEN)
    sts = pcr_set_stream_sample_rate(handle, DECIMATION)
    sts = pcr_set_stream_bandwidth(handle, 40.0e6)
    # We are using the API to retrieve the I/Q samples, and not Vita49 streaming
    sts = pcr_set_stream_mode(handle, PCRStreamMode.PCRStreamModeLocal)

    # Pattern is provided as interleaved I/Q samples
    # The pattern must be a multiple of 16 I/Q samples
    VSG_PATTERN_LEN = 16
    vsg_pattern = np.zeros(VSG_PATTERN_LEN, dtype=np.complex64)
    for i in range(VSG_PATTERN_LEN):
        vsg_pattern[i] = 1.0

    # Configure the VSG
    pcr_set_vsg_pattern(handle, vsg_pattern, VSG_PATTERN_LEN, PCRDataType.PCRDataType32fc)
    pcr_set_vsg_level(handle, -25.0)
    pcr_set_vsg_enabled(handle, True)

    # Start the measurements
    sts = pcr_initiate(handle)

    # Query the stream parameters
    status, channel_count, sample_rate, bandwidth = pcr_stream_parameters(handle).values()

    assert channel_count == 1
    assert sample_rate == EXPECTED_SAMPLE_RATE

    # Number of samples to query at a time
    N = 1_000_000
    iq = np.zeros((4, N)).astype(np.complex64)

    # Collect 1 second worth of data
    for i in range(50):
        pcr_stream_recv(handle, iq, N, False)

        # Do something with the data here..
        # For this example we will print off average power
        power = avg_power_dbm(iq[0], N)
        print(f'Power {power:.2f}')

    # Done, close device
    pcr_close_device(handle)


if __name__ == '__main__':
    pcr_example_vsg_and_rx()
