#include <cstdio>
#include <cstdlib>
#include <vector>

#include "sp_api.h"

// This example demonstrates how to interface the API for external triggers
//   during I/Q streaming.
// This example waits 1 second for an external trigger and if one is seen, then
//   collects 1 second of I/Q data.

void sp_example_iq_ext_trigger()
{
    int handle = -1;
    SpStatus status = spNoError;

    // Open device
    status = spOpenDevice(&handle);

    if(status != spNoError) {
        printf("Unable to open device\n");
        exit(-1);
    }

    // Configure the receiver for IQ acquisition
    spSetIQCenterFreq(handle, 2.45e9); // 2.45GHz
    spSetIQSampleRate(handle, 2); // 50 / 2 = 25MS/s IQ 
    spSetIQBandwidth(handle, 20.0e6); // 20MHz of bandwidth

    // Initialize the receiver with the above settings
    status = spConfigure(handle, spModeIQStreaming);
    if(status != spNoError) {
        printf("Unable to configure device\n");
        printf("%s\n", spGetErrorString(status));
        spCloseDevice(handle);
        exit(-1);
    }

    // Query the receiver IQ stream characteristics
    // Should match what we set earlier
    double actualSampleRate, actualBandwidth;
    spGetIQParameters(handle, &actualSampleRate, &actualBandwidth);

    // Allocate memory for complex samples, IQ pairs interleaved
    std::vector<float> iqCapture((int)actualSampleRate * 2);
    double trigger = -1;

    printf("Waiting for external trigger\n");

    // Wait N seconds for ext trigger
    const int searchTimeSec = 3; // Number of seconds to wait
    int waitSize = 16384;
    int waitCaptures = searchTimeSec * ((int)actualSampleRate / waitSize + 1);
    while(waitCaptures > 0) {
        spGetIQ(handle, &iqCapture[0], waitSize, &trigger, 1, 0, spFalse, 0, 0);

        if(trigger > 0) {
            printf("Found external trigger !\n");
            break;
        }

        waitCaptures--;
    }

    if(trigger > 0) {
        int iTrigPos = (int)trigger;
        // Found trigger, lets collect up to 1 second of data after the trigger
        // First grab all the samples after the trigger from the previous spGetIQ call
        int remaining = waitSize - iTrigPos;
        int samplesLeft = ((int)iqCapture.size() / 2) - remaining;
        memmove(&iqCapture[0], &iqCapture[iTrigPos * 2], remaining * 2 * sizeof(float));

        // Grab remaining samples
        spGetIQ(handle, &iqCapture[remaining * 2], samplesLeft, 0, 0, 0, spFalse, 0, 0);

        // At this point we have 1 second of IQ data after an external trigger is seen
        // The I/Q data is in the iqCapture buffer, with the first index being the first
        //   sample after the ext trigger.
    } else {
        printf("Did not find external trigger\n");
    }

    // Finished
    spCloseDevice(handle);
}