#include "pcr_v49_ingestor.h"

#include <iostream>
#include <cassert>
#include <vector>
#include <unordered_map>

// link with Ws2_32.lib
#pragma comment (lib, "Ws2_32.lib")

// This example illustrates how to ingest a V49 data stream that has been redirected from a PCR
// Included with this are all the data structures needed to extract V49 data from context and data packets

#define PC_ADDR ("192.168.2.2")
#define PCR_ADDR ("192.168.2.10")
#define PCR_PORT (4992)
#define UDP_QUEUE_SZ (8)

NetworkedDataTransfer::NetworkedDataTransfer(SOCKET socket, int slot) :
    socket(socket),
    slot(slot),
    dgramsInFlight(0),
    timeoutRequested(0),
    timedOut(false)
{
    for(int i = 0; i < NetXferSizes::DGRAMS_PER_XFER; i++) {
        inputBuf[i] = (uint8_t*)malloc(NetXferSizes::DGRAM_DATA_LEN);
        xferStatus.p[i] = inputBuf[i];
        xferStatus.xferLen[i] = NetXferSizes::DGRAM_DATA_LEN;
    }
    xferStatus.dgramsRequested = 0;

    // Setup overlapped I/O structs
    for(int i = 0; i < NetXferSizes::DGRAMS_PER_XFER; i++) {
        SecureZeroMemory(&overlapped[i], sizeof(WSAOVERLAPPED));
        overlapped[i].hEvent = WSACreateEvent();
        eventArray[i] = overlapped[i].hEvent;
    }
}

NetworkedDataTransfer::~NetworkedDataTransfer()
{
    for(int i = 0; i < NetXferSizes::DGRAMS_PER_XFER; i++) {
        free(inputBuf[i]);
    }
}

void NetworkedDataTransfer::BeginDataXfer(int dataGrams, int timeoutMs)
{
    dgramsInFlight = dataGrams;
    timeoutRequested = timeoutMs;

    for(int i = 0; i < dgramsInFlight; i++) {
        DWORD flags = 0;
        DWORD bytesReceived = 0;

        // Setup buffer to place xfer
        wsaBufs[i].len = NetXferSizes::DGRAM_DATA_LEN;
        wsaBufs[i].buf = (char*)xferStatus.p[i];

        int err = WSARecv(socket, &wsaBufs[i], WSA_BUFS, &bytesReceived,
            &flags, &overlapped[i], NULL);

        //   An error here is expected. If you pull the last error from WSAGetLasError
        //   you should see an WSA_IO_PENDING meaning that an overlapped operation has begun
        //   successfully. For now, don't bother checking.
    }

    xferStatus.dgramsRequested = dataGrams;
}

const XferStatus* NetworkedDataTransfer::FinishDataXfer()
{
    uint64_t bytesXferred = 0;

    // Wait for up to 64 dgrams at a time
    for(int d = 0; d < dgramsInFlight; ) {
        int left = dgramsInFlight - d;
        if(left > 64) {
            left = 64;
        }

        DWORD result = WSAWaitForMultipleEvents(left,
            eventArray + d, TRUE, timeoutRequested, TRUE);

        if(result == WSA_WAIT_TIMEOUT) {
            timedOut = true;
            xferStatus.timeout = true;
            closesocket(socket);
        }

        d += left;
    }

    for(int i = 0; i < dgramsInFlight; i++) {
        DWORD flags = 0;
        DWORD bytesReceived = 0;
        WSAGetOverlappedResult(socket, &overlapped[i], &bytesReceived, FALSE, &flags);

        bytesXferred += bytesReceived;
        xferStatus.xferLen[i] = bytesReceived;
    }

    dgramsInFlight = 0;

    return &xferStatus;
}

// Parse the V49 packets
void V49Ingestor::ParsePkts(const XferStatus *sts)
{
    PktInfo info;

    // Loop through all data grams
    for(int i = 0; i < sts->dgramsRequested; i++) {
        // parse the vita49 packet header
        uint32_t *pkt = (uint32_t*)sts->p[i];
        vrtSwapBytes(pkt, 5); // vita49 is big endian so we need to do a byte swap before doing things with the data

        // Packet counter
        int pktCounter = vrtGetDataPacketCount(pkt[0]);
        int pktSize = vrtGetDataPacketSize(pkt[0]);

        int streamID = pkt[1];

        uint64_t ns = vrtGetNsTimestamp(pkt[2], pkt[3], pkt[4]);
        // GPS to UTC, add 18 seconds
        uint64_t nsSinceEpoch = ns + 18e9;

        VRTPacketType pktType = (VRTPacketType)((pkt[0] & 0xF0000000) >> 28);

        // Debugging, store information about every packet.
        DWORD currentTime = GetTickCount();
        DWORD elapsedMs = currentTime - lastClear;
        double elapsedSec = (double)elapsedMs / 1000.0;

        if(lastClear == -1) {
            lastClear = currentTime;
        } 

        if((currentTime - lastClear) > 1000) {
            // Let's count data packets
            std::unordered_map<int, int> map;
            for(int i = 0; i < history.size(); i++) {
                if(history[i].type == VRTDataPacket) {
                    map[history[i].id] += 1;
                }
            }

            for(const std::pair<int, int> &p : map) {
                double msps = ((p.second * 2048.0) / elapsedSec) / 1.0e6;
                printf("ID: %d, %.2f MS/s\n", p.first, msps);
            }
            printf("\n");

            history.clear();
            lastClear = currentTime;
        }

        info.type = pktType;
        info.id = streamID;
        history.push_back(info);

        // Packet type
        switch(pktType) {
        case VRTDataPacket:
        {
            // Check packet counters
            if(lastPktCounter != -1) {
                uint8_t counterDelta = (pktCounter - lastPktCounter) & 0xF;
                if(counterDelta > 1) {
                    std::cout << " Packets are not continuous. Last Packet = "
                        << lastPktCounter 
                        << " Current Packet Counter = " 
                        << pktCounter << std::endl;
                }
            } 
            lastPktCounter = pktCounter;

            // Packet size for data packets should always be 2054
            if(pktSize != 2054) {
                std::cout << "Invalid IF Data Packet Size = " << pktSize << std::endl;
            }

            // Packet size should always correspond to the number of packets read
            if(sts->xferLen[i] != pktSize * sizeof(uint32_t)) {
                std::cout << "Transfer length and packet size mismatch\n Transfer Length = " << sts->xferLen[i] << " Packet Size = " << pktSize * sizeof(uint32_t) << std::endl;
            }

            // Parse trailer bits
            uint32_t trailer = pkt[pktSize - 1];
            vrtSwapBytes(&trailer, 1);

            uint32_t trailerIndicators = trailer & 0x000FFF00;
            // update transfer status based on trailer

            if(trailerIndicators > 0) {
                // Calibrated Time Indicator
                if(vrtGetBit(trailer, VRT_TIME_ENABLE)) {
                    if(vrtGetBit(trailer, VRT_TIME_INDICATOR)) {
                        //std::cout << "Time is not calibrated" << std::endl;
                    }
                }
                // Valid Data Indicator
                if(vrtGetBit(trailer, VRT_VALID_DATA_ENABLE)) {
                    if(!vrtGetBit(trailer, VRT_VALID_DATA_INDICATOR)) {
                        std::cout << "Data is invalid " << std::endl;
                    }
                }
                // Reference Lock Indicator
                if(vrtGetBit(trailer, VRT_REF_LOCK_ENABLE)) {
                    if(!vrtGetBit(trailer, VRT_REF_LOCK_INDICATOR)) {
                        std::cout << "LO is not locked" << std::endl;
                    }
                }
                // Over-Range Indicator
                if(vrtGetBit(trailer, VRT_OVER_RANGE_ENABLE)) {
                    if(vrtGetBit(trailer, VRT_OVER_RANGE_INDICATOR)) {
                        std::cout << "ADC overflow" << std::endl;
                    }
                }
                // Sample Loss Indicator
                if(vrtGetBit(trailer, VRT_SAMPLE_LOSS_ENABLE)) {
                    if(vrtGetBit(trailer, VRT_SAMPLE_LOSS_INDICATOR)) {
                        std::cout << "Samples lost" << std::endl;
                    }
                }
                // User defined, has trigger indicator, cache to search for trigger later
                if(vrtGetBit(trailer, VRT_HAS_TRIGGER_ENABLE)) {
                    uint32_t hasTriggerBits =
                        vrtGetBit(trailer, VRT_HAS_TRIGGER_INDICATOR);
                }
            }
            break;
        }

        case VRTContextPacket:
        {
            VrtContext vrtCntx;
            uint32_t *ctxtData = &pkt[5];
            vrtCntx.streamID = streamID;
            vrtCntx.pktCounter = pktCounter;
            vrtCntx.pktSize = pktSize;
            vrtCntx.contextPktsSinceLastChange += 1;
            vrtCntx.lastContextFields = vrtGetContextFields(ctxtData);
            break;
        }

        default:
        {
            // Should never get here!
            std::cout << "Invalid Packet Type = " << pktType << std::endl;
        }
        }
    }
}

int main()
{
    // Data xfer variables
    NetworkedDataTransfer* dataXfers[NetXferSizes::XFER_SLOTS];
    int dgramsPerXfer = NetXferSizes::DGRAMS_PER_XFER;
    // Configure WSA sockert for overlapped reads
    // the overlapped reads are essential for handlings the 6.4gbps the pcr is capable of streaming
    SOCKET socket = INVALID_SOCKET;
    uint64_t packetsIngested;

    int err = 0;
    WSAData wsaData;
    err = WSAStartup(MAKEWORD(2, 2), &wsaData);
    if(err) {
        return false;
    }

    // Receive side setup
    socket = WSASocketA(AF_INET, SOCK_DGRAM, IPPROTO_UDP, NULL, 0, WSA_FLAG_OVERLAPPED);
    if(socket == INVALID_SOCKET) {
        return false;
    }

    BOOL reuseAddr = TRUE;
    err = setsockopt(socket, SOL_SOCKET, SO_REUSEADDR, (char*)&reuseAddr, sizeof(reuseAddr));
    if(err) {
        return false;
    }

    struct sockaddr_in hostAddr;
    hostAddr.sin_addr.s_addr = inet_addr(PC_ADDR);
    hostAddr.sin_family = AF_INET;
    hostAddr.sin_port = htons(PCR_PORT);

    err = bind(socket, (sockaddr*)&hostAddr, sizeof(hostAddr));
    if(err) {
        return false;
    }

    sockaddr_in deviceAddr;
    deviceAddr.sin_addr.s_addr = inet_addr(PCR_ADDR);
    deviceAddr.sin_family = AF_INET;
    deviceAddr.sin_port = htons(PCR_PORT);

    err = connect(socket, (sockaddr*)&deviceAddr, sizeof(deviceAddr));
    if(err) {
        return false;
    }

    // Create transfer classes for overlapped reads
    for(int i = 0; i < NetXferSizes::XFER_SLOTS; i++) {
        dataXfers[i] = new NetworkedDataTransfer(socket, i);
    }

    printf("Waiting for data\n");

    V49Ingestor ingestor;

    // Start data xfers
    for(int i = 0; i < UDP_QUEUE_SZ; i++) {
        dataXfers[i]->BeginDataXfer(dgramsPerXfer, WSA_INFINITE);
    }

    int slotIndex = 0;
    while(1) {
        const XferStatus* xferSts = dataXfers[slotIndex]->FinishDataXfer();

        if(!xferSts->Failed()) {
            // If data is successfully recieved parse the data as V49 packets
            ingestor.ParsePkts(xferSts);
        } else {
            break;
        }

        // Requeue the transfer
        dataXfers[slotIndex]->BeginDataXfer(dgramsPerXfer, WSA_INFINITE);

        if(++slotIndex >= UDP_QUEUE_SZ) {
            slotIndex = 0;
        }
    }
}
