"""
Demo of reading Sceptre LiveStream from network and plotting spectrum

This is just a quick demo to show how to plot the spectrum of a Sceptre LiveStream (either IF or Spectral data).
"""
import matplotlib.pyplot as plt
import numpy as np
from sceptrelivestream import LiveStream, PSDEngine, IFFrame, SpectralFrame, Timestamp

# UDP port to listen for LiveStream on
port = 12000
# FFT size to use if we need to calculate our own PSD
nfft = 4096
# Number of spectral frames to average if we need to calculate our own PSD
navg = 4

# Create livestream object
livestream = LiveStream('', port)

# Create PSDEngine that will be used if we receive time-domain data
psd = PSDEngine(nfft, navg)

# Initialize plots
fig, ax = plt.subplots()
line = None
plt.grid(True)
plt.show(block=False)
ax.set_ylim(-100.0, 100.0)

last_eph_time = Timestamp()

# Loop forever
while True:
    # Read the next VRT packet
    pkt = livestream.read()
    # Just continue the loop and read the next packet if we didn't get a frame of data
    if pkt is None or pkt.frame is None:
        continue

    # Handle the frame based on whether it is an IFFrame or SpectralFrame
    frame = pkt.frame
    psd_frame: SpectralFrame | None = None
    if isinstance(frame, IFFrame):
        # If this a IFFrame, we need to compute the PSD using the PSDEngine
        psd_frames = psd.process(frame)
        if len(psd_frames) > 0:
            # We got at least one output spectral frame.  Just plot the most recent.
            psd_frame = psd_frames[-1]
    elif isinstance(frame, SpectralFrame):
        # This is already a spectral frame, just plot it
        psd_frame = frame

    # Print ephemeris updates to the console (only print new updates)
    if pkt.context is not None and pkt.context.ephemeris is not None:
        eph = pkt.context.ephemeris
        if eph.timestamp != last_eph_time:
            print(f'ephemeris: time={eph.timestamp}  x,y,z={eph.position}')
            last_eph_time = eph.timestamp

    # If we were able to get a PSD frame, plot it
    if psd_frame is not None:
        psd_db = psd_frame.data

        # Get the x (frequency) values for the plot
        xs = psd_frame.frequency_start + np.arange(0, len(psd_frame.data)) * psd_frame.frequency_delta

        # Update the live data
        if line is None:
            line, = ax.plot(xs, psd_db)
        else:
            line.set_ydata(psd_db)
            line.set_xdata(xs)

        # Show the timestamp in the title of the plot
        plt.title(f'{frame.timestamp}')

        # Update the plot display
        plt.draw()
        fig.canvas.update()
        fig.canvas.flush_events()
