from enum import IntEnum

from ctypes import CDLL
from ctypes import Structure
from ctypes import POINTER
from ctypes import byref
from ctypes import c_int
from ctypes import c_int64
from ctypes import c_uint16
from ctypes import c_uint32
from ctypes import c_uint64
from ctypes import c_float
from ctypes import c_double
from ctypes import c_char
from ctypes import c_char_p

import numpy as np

# Load the library
pcrlib = CDLL('pcrdevice/pcr_api.dll') # or 'libpcr_api.so' on Linux


# ---------------------------------- Defines -----------------------------------

#  Maximum number of device that can be interfaced by the API
PCR_MAX_DEVICES = 4

#  Convenience host address
PCR_DEFAULT_HOST_ADDR = '192.168.2.2'

# Default PCR IP address
PCR_DEFAULT_DEVICE_ADDR = '192.168.2.10'
# Default PCR port
PCR_DEFAULT_PORT = 51665
# Default streaming port
PCR_DEFAULT_STREAM_PORT = 4991

# Used in pcrSetChannelConfig to indicate no sweep channel
PCR_SWEEP_CHANNEL_DISABLED = -1

# Indicates the API should use the reference level to control sensitivity. Default
PCR_AUTO_ATTEN = -1
# Valid attenuation values between [0 6] or -1 for auto
PCR_MAX_ATTEN = 6

# Minimum configurable frequency
PCR_MIN_FREQ = 100.0e3
# Maximum configurable frequency
PCR_MAX_FREQ = 20.0e9
# Minimum configurable sweep span
PCR_SWEEP_MIN_SPAN = 100.0
# Minimum configurable sweep RBW
PCR_SWEEP_MIN_RBW = 100.0
# Maximum configurable sweep RBW
PCR_SWEEP_MAX_RBW = 10.0e6
# Minimum configurable sweep time
PCR_MIN_SWEEP_TIME = 1.0e-6
# Maximum configurable sweep time
PCR_MAX_SWEEP_TIME = 1.0

# Minimum configurable reference level for sweeps and I/Q streaming
PCR_MIN_REF_LVL = -130.0
# Maximum configurable reference level for sweeps and I/Q streaming
PCR_MAX_REF_LVL = 20.0

# The maximum I/Q stream sample rate decimation = 1
PCR_STREAM_SR = 50.0e6
# Minimum configurable I/Q stream decimation
PCR_STREAM_MIN_DECIMATION = 1
# Maximum configurable I/Q stream decimation
PCR_STREAM_MAX_DECIMATION = 256
# Minimum configurable I/Q stream bandwidth
PCR_STREAM_MIN_BW = 20.0e3
# Maximum configurable I/Q stream bandwidth
PCR_STREAM_MAX_BW = 40.0e6

# Minimum Vita49 stream mode context interval
PCR_MIN_CNTX_INTERVAL = 3
# Maximum Vita49 stream mode context interval
PCR_MAX_CNTX_INTERVAL = 4095

# Minimum VSG output level = dBm
PCR_VSG_MIN_LEVEL = -80.0
# Maximum VSG output level = dBm
PCR_VSG_MAX_LEVEL = 10.0
# Maximum VSG pattern buffer length = complex samples
PCR_VSG_MAX_BUF_LEN = 4096

# Minimum fan setpoint in C
PCR_MIN_FAN_SETPOINT = 5.0
# Maximum fan setpoint in C
PCR_MAX_FAN_SETPOINT = 60.0

# Enums
class C_Enum(IntEnum):
    def __init__(self, value):
        self._as_parameter = int(value)

    @classmethod
    def from_param(cls, obj):
        return int(obj)

class PCRBool(C_Enum):
    # False
    PCRBoolFalse = 0
    # True
    PCRBoolTrue = 1

class PCRDataType(C_Enum):
    # 16-bit interleaved complex integers.
    PCRDataType16sc = 0
    # 32-bit interleaved complex floating point.
    PCRDataType32fc = 1

class PCRReference(C_Enum):
    # Use the internal 10MHz reference clock.
    PCRReferenceInternal = 0
    # Use 10MHz reference connected to the reference input RF port.
    PCRReferenceExternal = 1

class PCRGPSState(C_Enum):
    # No GPS lock detected.
    PCRGPSStateNotPresent = 0
    # GPS is locked.
    PCRGPSStateLocked = 1
    # GPS is locked and disciplining the system.
    # Disciplining must be explictely enabled to get to this state.
    PCRGPSStateDisciplined = 2

class PCREndianness(C_Enum):
    # Big endian default.
    PCREndiannessBig = 0
    # Little endian.
    PCREndiannessLittle = 1

class PCRSweepDetector(C_Enum):
    # Average detector.
    PCRSweepDetectorAvg = 0
    # Min hold detector.
    PCRSweepDetectorMin = 1
    # Max hold detector.
    PCRSweepDetectorMax = 2

class PCRSweepVideoUnits(C_Enum):
    # dBm
    PCRSweepVideoUnitsLog = 0
    # Linear voltage
    PCRSweepVideoUnitsVoltage = 1
    # Linear power
    PCRSweepVideoUnitsPower = 2
    # No VBW processing
    PCRSweepVideoUnitsSample = 3

class PCRSweepScale(C_Enum):
    # dBm
    PCRSweepScaleLog = 0
    # mV
    PCRSweepScaleLin = 1
    # dBFS no corrections
    PCRSweepScaleFullScale = 2

class PCRSweepWindow(C_Enum):
    # SRS flattop
    PCRSweepWindowFlattop = 0
    # Nutall window
    PCRSweepWindowNutall = 1
    # Gaussian 6dB window used for EMC measurements and CISPR compatibility.
    PCRSweepWindowGaussian6dB = 2

class PCRTriggerEdge(C_Enum):
    # Rising edge triggering.
    PCRTriggerEdgeRising = 0
    # Falling edge triggering.
    PCRTriggerEdgeFalling = 1

class PCRStreamMode(C_Enum):
    PCRStreamModeLocal = 0
    PCRStreamModeVRT = 1

class PCRStatus(C_Enum):
    # Returned when the device detects issues on ADC clocks and sample alignment.
    # Indicates possible hardware failure. Contact Signal Hound.
    PCRStatusADCErr = -12
    # Returned when performing an invalid operation on a multi-receiver/paired system.
    PCRStatusPairedErr = -11
    # Returned when the device detects framing issue on measurement data
    # Measurement results are likely invalid. Device should be preset/power
    # cycled.
    PCRStatusSyncErr = -10
    # Internal error.
    PCRStatusIOErr = -9
    # One or more parameters provided were invalid.
    PCRStatusInvalidParameterErr = -8
    # Attempting to perform an operation that cannot currently be performed.
    # Often the result of trying to do something while the device is currently
    # making measurements or not in an idle state.
    PCRStatusInvalidConfigurationErr = -7
    # Out of memory.
    PCRStatusMemoryErr = -6
    # No more devices can be interfaced. A device needs to be closed before
    # another can be connected.
    PCRStatusMaxDevicesConnectedErr = -5
    # Device index specified is invalid.
    PCRStatusInvalidDeviceErr = -4
    # One or more required pointer parameters are NULL.
    PCRStatusNullPtrErr = -3
    # Device disconnected. Usually the result of network errors or UDP packet loss.
    PCRStatusDeviceConnectionLostErr = -2
    # Unable to open device.
    PCRStatusDeviceNotFoundErr = -1

    # Function returned successfully.
    PCRStatusNoError = 0

    # One or more parameters were clamped to a valid range.
    PCRStatusSettingClamped = 1
    # Measurement includes data subject to an ADC overload (clipping/compression)
    PCRStatusADCOverflow = 2
    # Measurement uncalibrated. Overrides ADC overflow. Often the result of data loss.
    PCRStatusUncalData = 3
    # Temperature drift has occured since last configured.
    # Measurement error may be present.
    # Reinitialize the measurement to resolve.
    PCRStatusTempDriftWarning = 4
    # Indicates internal buffer wrap due to network throughput.
    # Data may be invalid
    PCRStatusBufferOverflow = 5
    # Calibration data potentially corrupt.
    PCRStatusInvalidCalData = 6


# Structs
class PCRDeviceDiagnostics(Structure):
    _fields_ = [
        # Device voltage in V
        ('voltage', c_float),
        # Input current in A
        ('currentInput', c_float),
        # OCXO current in A
        ('currentOCXO', c_float),
        # 5.8V power supply current in A
        ('current58', c_float),
        # Internal FPGA temperature in C
        ('tempFPGA', c_float),
        # Motherboard temp in C
        ('tempMB', c_float),
        # Temperature on RF board 1 in C
        ('tempRFBoard1', c_float),
        # Temperature on RF board 2 in C
        ('tempRFBoard2', c_float),
        # SFP+ temperature in C
        ('sfpTemp', c_float),
        # SFP+ Tx power in mW
        ('sfpTxPower', c_float),
        # SFP+ Rx power in mW
        ('sfpRxPower', c_float)
    ]


# Function prototypes
pcrConnectDevice = pcrlib.pcrConnectDevice
pcrConnectDevice.argtypes = [POINTER(c_int), POINTER(c_char), POINTER(c_char), c_uint16]
pcrConnectDevice.restype = PCRStatus

pcrPairDevice = pcrlib.pcrPairDevice
pcrPairDevice.argtypes = [c_int, POINTER(c_char), POINTER(c_char), c_uint16]
pcrPairDevice.restype = PCRStatus

pcrCloseDevice = pcrlib.pcrCloseDevice
pcrCloseDevice.argtypes = [c_int]
pcrCloseDevice.restype = PCRStatus

pcrGetSerialNumber = pcrlib.pcrGetSerialNumber
pcrGetSerialNumber.argtypes = [c_int, POINTER(c_int)]
pcrGetSerialNumber.restype = PCRStatus

pcrGetFirmwareVersion = pcrlib.pcrGetFirmwareVersion
pcrGetFirmwareVersion.argtypes = [c_int, POINTER(c_int), POINTER(c_int), POINTER(c_int)]
pcrGetFirmwareVersion.restype = PCRStatus

pcrGetCalDate = pcrlib.pcrGetCalDate
pcrGetCalDate.argtypes = [c_int, POINTER(c_uint32)]
pcrGetCalDate.restype = PCRStatus

pcrGetDeviceDiagnostics = pcrlib.pcrGetDeviceDiagnostics
pcrGetDeviceDiagnostics.argtypes = [c_int, POINTER(c_float), POINTER(c_float), POINTER(c_float)]
pcrGetDeviceDiagnostics.restype = PCRStatus

pcrGetFullDeviceDiagnostics = pcrlib.pcrGetFullDeviceDiagnostics
pcrGetFullDeviceDiagnostics.argtypes = [c_int, POINTER(PCRDeviceDiagnostics)]
pcrGetFullDeviceDiagnostics.restype = PCRStatus

pcrSetLowPowerState = pcrlib.pcrSetLowPowerState
pcrSetLowPowerState.argtypes = [c_int]
pcrSetLowPowerState.restype = PCRStatus

pcrSetReference = pcrlib.pcrSetReference
pcrSetReference.argtypes = [c_int, PCRReference]
pcrSetReference.restype = PCRStatus

pcrGetReference = pcrlib.pcrGetReference
pcrGetReference.argtypes = [c_int, POINTER(c_int)]
pcrGetReference.restype = PCRStatus

pcrSetGPSTimebaseUpdate = pcrlib.pcrSetGPSTimebaseUpdate
pcrSetGPSTimebaseUpdate.argtypes = [c_int, PCRBool]
pcrSetGPSTimebaseUpdate.restype = PCRStatus

pcrGetGPSTimebaseUpdate = pcrlib.pcrGetGPSTimebaseUpdate
pcrGetGPSTimebaseUpdate.argtypes = [c_int, POINTER(c_int)]
pcrGetGPSTimebaseUpdate.restype = PCRStatus

pcrGetGPSHoldoverInfo = pcrlib.pcrGetGPSHoldoverInfo
pcrGetGPSHoldoverInfo.argtypes = [c_int, POINTER(c_int), POINTER(c_uint64)]
pcrGetGPSHoldoverInfo.restype = PCRStatus

pcrGetGPSState = pcrlib.pcrGetGPSState
pcrGetGPSState.argtypes = [c_int, POINTER(c_int)]
pcrGetGPSState.restype = PCRStatus

pcrGetGPSInfo = pcrlib.pcrGetGPSInfo
pcrGetGPSInfo.argtypes = [c_int, PCRBool, POINTER(c_int), POINTER(c_int64), POINTER(c_double), POINTER(c_double), POINTER(c_double)]
pcrGetGPSInfo.restype = PCRStatus

pcrWriteMessageToGPS = pcrlib.pcrWriteMessageToGPS
pcrWriteMessageToGPS.argtypes = [c_int, c_uint32, c_uint64, c_int]
pcrWriteMessageToGPS.restype = PCRStatus

pcrSetFanEnabled = pcrlib.pcrSetFanEnabled
pcrSetFanEnabled.argtypes = [c_int, PCRBool]
pcrSetFanEnabled.restype = PCRStatus

pcrGetFanEnabled = pcrlib.pcrGetFanEnabled
pcrGetFanEnabled.argtypes = [c_int, POINTER(c_int)]
pcrGetFanEnabled.restype = PCRStatus

pcrSetFanSetPoint = pcrlib.pcrSetFanSetPoint
pcrSetFanSetPoint.argtypes = [c_int, c_float]
pcrSetFanSetPoint.restype = PCRStatus

pcrGetFanSetPoint = pcrlib.pcrGetFanSetPoint
pcrGetFanSetPoint.argtypes = [c_int, POINTER(c_float)]
pcrGetFanSetPoint.restype = PCRStatus

pcrGetChannelCount = pcrlib.pcrGetChannelCount
pcrGetChannelCount.argtypes = [c_int, POINTER(c_int)]
pcrGetChannelCount.restype = PCRStatus

pcrSetChannelConfig = pcrlib.pcrSetChannelConfig
pcrSetChannelConfig.argtypes = [c_int, POINTER(c_int), c_int]
pcrSetChannelConfig.restype = PCRStatus

pcrGetChannelConfig = pcrlib.pcrGetChannelConfig
pcrGetChannelConfig.argtypes = [c_int, POINTER(c_int), POINTER(c_int)]
pcrGetChannelConfig.restype = PCRStatus

pcrSetChannelShared = pcrlib.pcrSetChannelShared
pcrSetChannelShared.argtypes = [c_int, c_int, PCRBool]
pcrSetChannelShared.restype = PCRStatus

pcrGetChannelShared = pcrlib.pcrGetChannelShared
pcrGetChannelShared.argtypes = [c_int, c_int, POINTER(c_int)]
pcrGetChannelShared.restype = PCRStatus

pcrSetChannelFreq = pcrlib.pcrSetChannelFreq
pcrSetChannelFreq.argtypes = [c_int, c_int, c_double]
pcrSetChannelFreq.restype = PCRStatus

pcrGetChannelFreq = pcrlib.pcrGetChannelFreq
pcrGetChannelFreq.argtypes = [c_int, c_int, POINTER(c_double)]
pcrGetChannelFreq.restype = PCRStatus

pcrSetChannelPhaseOffset = pcrlib.pcrSetChannelPhaseOffset
pcrSetChannelPhaseOffset.argtypes = [c_int, c_int, c_double]
pcrSetChannelPhaseOffset.restype = PCRStatus

pcrGetChannelPhaseOffset = pcrlib.pcrGetChannelPhaseOffset
pcrGetChannelPhaseOffset.argtypes = [c_int, c_int, POINTER(c_double)]
pcrGetChannelPhaseOffset.restype = PCRStatus

pcrSetChannelDelay = pcrlib.pcrSetChannelDelay
pcrSetChannelDelay.argtypes = [c_int, c_int, c_double]
pcrSetChannelDelay.restype = PCRStatus

pcrGetChannelDelay = pcrlib.pcrGetChannelDelay
pcrGetChannelDelay.argtypes = [c_int, c_int, POINTER(c_double)]
pcrGetChannelDelay.restype = PCRStatus

pcrSetSharedFreq = pcrlib.pcrSetSharedFreq
pcrSetSharedFreq.argtypes = [c_int, c_double]
pcrSetSharedFreq.restype = PCRStatus

pcrGetSharedFreq = pcrlib.pcrGetSharedFreq
pcrGetSharedFreq.argtypes = [c_int, POINTER(c_double)]
pcrGetSharedFreq.restype = PCRStatus

pcrSetStreamDataType = pcrlib.pcrSetStreamDataType
pcrSetStreamDataType.argtypes = [c_int, PCRDataType]
pcrSetStreamDataType.restype = PCRStatus

pcrSetStreamFullScale = pcrlib.pcrSetStreamFullScale
pcrSetStreamFullScale.argtypes = [c_int, PCRBool]
pcrSetStreamFullScale.restype = PCRStatus

pcrSetStreamRefLevel = pcrlib.pcrSetStreamRefLevel
pcrSetStreamRefLevel.argtypes = [c_int, c_double]
pcrSetStreamRefLevel.restype = PCRStatus

pcrSetStreamAtten = pcrlib.pcrSetStreamAtten
pcrSetStreamAtten.argtypes = [c_int, c_int]
pcrSetStreamAtten.restype = PCRStatus

pcrSetStreamSampleRate = pcrlib.pcrSetStreamSampleRate
pcrSetStreamSampleRate.argtypes = [c_int, c_int]
pcrSetStreamSampleRate.restype = PCRStatus

pcrSetStreamBandwidth = pcrlib.pcrSetStreamBandwidth
pcrSetStreamBandwidth.argtypes = [c_int, c_double]
pcrSetStreamBandwidth.restype = PCRStatus

pcrSetStreamTriggerEdge = pcrlib.pcrSetStreamTriggerEdge
pcrSetStreamTriggerEdge.argtypes = [c_int, PCRTriggerEdge]
pcrSetStreamTriggerEdge.restype = PCRStatus

pcrSetStreamMode = pcrlib.pcrSetStreamMode
pcrSetStreamMode.argtypes = [c_int, PCRStreamMode]
pcrSetStreamMode.restype = PCRStatus

pcrSetStreamLocalAddr = pcrlib.pcrSetStreamLocalAddr
pcrSetStreamLocalAddr.argtypes = [c_int, c_uint16]
pcrSetStreamLocalAddr.restype = PCRStatus

pcrSetStreamVrtAddr = pcrlib.pcrSetStreamVrtAddr
pcrSetStreamVrtAddr.argtypes = [c_int, c_int, POINTER(c_char), c_uint16, c_uint64]
pcrSetStreamVrtAddr.restype = PCRStatus

pcrSetStreamVrtChannelID = pcrlib.pcrSetStreamVrtChannelID
pcrSetStreamVrtChannelID.argtypes = [c_int, c_int, c_int]
pcrSetStreamVrtChannelID.restype = PCRStatus

pcrSetStreamVrtCntxInterval = pcrlib.pcrSetStreamVrtCntxInterval
pcrSetStreamVrtCntxInterval.argtypes = [c_int, PCRBool, c_int]
pcrSetStreamVrtCntxInterval.restype = PCRStatus

pcrSetStreamVrtEndianness = pcrlib.pcrSetStreamVrtEndianness
pcrSetStreamVrtEndianness.argtypes = [c_int, PCREndianness]
pcrSetStreamVrtEndianness.restype = PCRStatus

pcrSetSweepRefLevel = pcrlib.pcrSetSweepRefLevel
pcrSetSweepRefLevel.argtypes = [c_int, c_double]
pcrSetSweepRefLevel.restype = PCRStatus

pcrSetSweepAtten = pcrlib.pcrSetSweepAtten
pcrSetSweepAtten.argtypes = [c_int, c_int]
pcrSetSweepAtten.restype = PCRStatus

pcrSetSweepCenterSpan = pcrlib.pcrSetSweepCenterSpan
pcrSetSweepCenterSpan.argtypes = [c_int, c_double, c_double]
pcrSetSweepCenterSpan.restype = PCRStatus

pcrSetSweepStartStop = pcrlib.pcrSetSweepStartStop
pcrSetSweepStartStop.argtypes = [c_int, c_double, c_double]
pcrSetSweepStartStop.restype = PCRStatus

pcrSetSweepCoupling = pcrlib.pcrSetSweepCoupling
pcrSetSweepCoupling.argtypes = [c_int, c_double, c_double, c_double]
pcrSetSweepCoupling.restype = PCRStatus

pcrSetSweepDetector = pcrlib.pcrSetSweepDetector
pcrSetSweepDetector.argtypes = [c_int, PCRSweepDetector, PCRSweepVideoUnits]
pcrSetSweepDetector.restype = PCRStatus

pcrSetSweepScale = pcrlib.pcrSetSweepScale
pcrSetSweepScale.argtypes = [c_int, PCRSweepScale]
pcrSetSweepScale.restype = PCRStatus

pcrSetSweepWindow = pcrlib.pcrSetSweepWindow
pcrSetSweepWindow.argtypes = [c_int, PCRSweepWindow]
pcrSetSweepWindow.restype = PCRStatus

pcrSetVSGEnabled = pcrlib.pcrSetVSGEnabled
pcrSetVSGEnabled.argtypes = [c_int, PCRBool]
pcrSetVSGEnabled.restype = PCRStatus

pcrSetVSGLevel = pcrlib.pcrSetVSGLevel
pcrSetVSGLevel.argtypes = [c_int, c_double]
pcrSetVSGLevel.restype = PCRStatus

pcrSetVSGPattern = pcrlib.pcrSetVSGPattern
pcrSetVSGPattern.argtypes = [c_int, np.ctypeslib.ndpointer(dtype=np.uintp), c_int, PCRDataType]
pcrSetVSGPattern.restype = PCRStatus

pcrInitiate = pcrlib.pcrInitiate
pcrInitiate.argtypes = [c_int]
pcrInitiate.restype = PCRStatus

pcrAbort = pcrlib.pcrAbort
pcrAbort.argtypes = [c_int]
pcrAbort.restype = PCRStatus

pcrStreamParameters = pcrlib.pcrStreamParameters
pcrStreamParameters.argtypes = [c_int, POINTER(c_int), POINTER(c_double), POINTER(c_double)]
pcrStreamParameters.restype = PCRStatus

pcrStreamCorrection = pcrlib.pcrStreamCorrection
pcrStreamCorrection.argtypes = [c_int, c_int, POINTER(c_float)]
pcrStreamCorrection.restype = PCRStatus

pcrStreamRecv = pcrlib.pcrStreamRecv
pcrStreamRecv.argtypes = [c_int, np.ctypeslib.ndpointer(dtype=np.uintp, ndim=1, flags='C'), c_int, POINTER(c_int64),
                          PCRBool, POINTER(c_int), POINTER(c_int), POINTER(c_double), c_int]
pcrStreamRecv.restype = PCRStatus

pcrSweepGetParameters = pcrlib.pcrSweepGetParameters
pcrSweepGetParameters.argtypes = [c_int, POINTER(c_double), POINTER(c_double), POINTER(c_double), POINTER(c_double), POINTER(c_int)]
pcrSweepGetParameters.restype = PCRStatus

pcrGetSweep = pcrlib.pcrGetSweep
pcrGetSweep.argtypes = [c_int, np.ctypeslib.ndpointer(np.float32, ndim=1, flags='C')]
pcrGetSweep.restype = PCRStatus

pcrSweepStart = pcrlib.pcrSweepStart
pcrSweepStart.argtypes = [c_int]
pcrSweepStart.restype = PCRStatus

pcrSweepFinish = pcrlib.pcrSweepFinish
pcrSweepFinish.argtypes = [c_int, np.ctypeslib.ndpointer(np.float32, ndim=1, flags='C')]
pcrSweepFinish.restype = PCRStatus

pcrNetworkConfigGetDeviceList = pcrlib.pcrNetworkConfigGetDeviceList
pcrNetworkConfigGetDeviceList.argtypes = [POINTER(c_int), POINTER(c_int)]
pcrNetworkConfigGetDeviceList.restype = PCRStatus

pcrNetworkConfigOpenDevice = pcrlib.pcrNetworkConfigOpenDevice
pcrNetworkConfigOpenDevice.argtypes = [POINTER(c_int), c_int]
pcrNetworkConfigOpenDevice.restype = PCRStatus

pcrNetworkConfigCloseDevice = pcrlib.pcrNetworkConfigCloseDevice
pcrNetworkConfigCloseDevice.argtypes = [c_int]
pcrNetworkConfigCloseDevice.restype = PCRStatus

pcrNetworkConfigGetMAC = pcrlib.pcrNetworkConfigGetMAC
pcrNetworkConfigGetMAC.argtypes = [c_int, POINTER(c_char)]
pcrNetworkConfigGetMAC.restype = PCRStatus

pcrNetworkConfigSetIP = pcrlib.pcrNetworkConfigSetIP
pcrNetworkConfigSetIP.argtypes = [c_int, POINTER(c_char), PCRBool]
pcrNetworkConfigSetIP.restype = PCRStatus

pcrNetworkConfigGetIP = pcrlib.pcrNetworkConfigGetIP
pcrNetworkConfigGetIP.argtypes = [c_int, POINTER(c_char)]
pcrNetworkConfigGetIP.restype = PCRStatus

pcrNetworkConfigSetPort = pcrlib.pcrNetworkConfigSetPort
pcrNetworkConfigSetPort.argtypes = [c_int, c_int, PCRBool]
pcrNetworkConfigSetPort.restype = PCRStatus

pcrNetworkConfigGetPort = pcrlib.pcrNetworkConfigGetPort
pcrNetworkConfigGetPort.argtypes = [c_int, POINTER(c_int)]
pcrNetworkConfigGetPort.restype = PCRStatus

pcrGetAPIVersion = pcrlib.pcrGetAPIVersion
pcrGetAPIVersion.restype = c_char_p

pcrGetErrorString = pcrlib.pcrGetErrorString
pcrGetErrorString.argtypes = [PCRStatus]
pcrGetErrorString.restype = c_char_p


# ---------------------------------- Utility ----------------------------------

def error_check(func):
    def print_status_if_error(*args, **kwargs):
        return_vars = func(*args, **kwargs)
        if 'status' not in return_vars.keys():
            return return_vars
        status = return_vars['status']
        if status != 0:
            print (f'{'Error' if status < 0 else 'Warning'} {status}: {pcr_get_error_string(status)} in {func.__name__}()')
        if status < 0:
            exit()
        return return_vars
    return print_status_if_error

def to_pcrbool(b: bool) -> PCRBool:
    return PCRBool.PCRBoolTrue if b else PCRBool.PCRBoolFalse

def from_pcrbool(sm_b: PCRBool) -> bool:
    return True if sm_b == PCRBool.PCRBoolTrue else False


# --------------------------------- Functions ---------------------------------

@error_check
def pcr_connect_device(host_addr: str, device_addr: str, port: int) -> dict[str, any]:
    handle = c_int(-1)

    status = pcrConnectDevice(byref(handle), host_addr.encode(), device_addr.encode(), port)

    return {
        'status': status,

        'handle': handle.value
    }

@error_check
def pcr_pair_device(handle: int, host_addr: str, device_addr: str, port: int) -> dict[str, any]:
    status = pcrPairDevice(handle, host_addr, device_addr, port)

    return {
        'status': status
    }

@error_check
def pcr_close_device(handle: int) -> dict[str, any]:
    status = pcrCloseDevice(handle)

    return {
        'status': status
    }

@error_check
def pcr_get_serial_number(handle: int) -> dict[str, any]:
    serial = c_int(-1)

    status = pcrGetSerialNumber(handle, byref(serial))

    return {
        'status': status,

        'serial': serial.value
    }

@error_check
def pcr_get_firmware_version(handle: int) -> dict[str, any]:
    major = c_int(-1)
    minor = c_int(-1)
    revision = c_int(-1)

    status = pcrGetFirmwareVersion(handle, byref(major), byref(minor), byref(revision))

    return {
        'status': status,

        'major': major.value,
        'minor': minor.value,
        'revision': revision.value
    }

@error_check
def pcr_get_cal_date(handle: int) -> dict[str, any]:
    last_cal_date = c_uint32(0)

    status = pcrGetCalDate(handle, byref(last_cal_date))

    return {
        'status': status,

        'last_cal_date': last_cal_date.value
    }

@error_check
def pcr_get_device_diagnostics(handle: int) -> dict[str, any]:
    voltage = c_float(-1)
    current = c_float(-1)
    temperature = c_float(-1)

    status = pcrGetDeviceDiagnostics(handle, byref(voltage), byref(current), byref(temperature))

    return {
        'status': status,

        'voltage': voltage.value,
        'current': current.value,
        'temperature': temperature.value
    }

@error_check
def pcr_get_full_device_diagnostics(handle: int) -> dict[str, any]:
    diagnostics = PCRDeviceDiagnostics()

    status = pcrGetFullDeviceDiagnostics(handle, byref(diagnostics))

    return {
        'status': status,

        'diagnostics': diagnostics
    }

@error_check
def pcr_set_low_power_state(handle: int) -> dict[str, any]:
    status = pcrSetLowPowerState(handle)

    return {
        'status': status
    }

@error_check
def pcr_set_reference(handle: int, reference: PCRReference) -> dict[str, any]:
    status = pcrSetReference(handle, reference)

    return {
        'status': status
    }

@error_check
def pcr_get_reference(handle: int) -> dict[str, any]:
    reference = c_int(PCRReference.PCRReferenceExternal)

    status = pcrGetReference(handle, byref(reference))

    return {
        'status': status,

        'reference': PCRReference(reference.value)
    }

@error_check
def pcr_set_gps_timebase_update(handle: int, enabled: bool) -> dict[str, any]:
    status = pcrSetGPSTimebaseUpdate(handle, to_pcrbool(enabled))

    return {
        'status': status
    }

@error_check
def pcr_get_gps_timebase_update(handle: int) -> dict[str, any]:
    enabled = c_int(-1)

    status = pcrGetGPSTimebaseUpdate(handle, byref(enabled))

    return {
        'status': status,

        'enabled': from_pcrbool(enabled.value)
    }

@error_check
def pcr_get_gps_holdover_info(handle: int) -> dict[str, any]:
    using_gps_holdover = c_int(-1)
    last_holdover_time = c_uint64(0)

    status = pcrGetGPSHoldoverInfo(handle, byref(using_gps_holdover), byref(last_holdover_time))

    return {
        'status': status,

        'using_gps_holdover': from_pcrbool(using_gps_holdover.value),
        'last_holdover_time': last_holdover_time.value
    }

@error_check
def pcr_get_gps_state(handle: int) -> dict[str, any]:
    state = c_int(PCRGPSState.PCRGPSStateNotPresent)

    status = pcrGetGPSState(handle, byref(state))

    return {
        'status': status,

        'state': PCRGPSState(state.value)
    }

@error_check
def pcr_get_gps_info(handle: int, refresh: bool) -> dict[str, any]:
    updated = c_int(-1)
    sec_since_epoch = c_int64(-1)
    latitude = c_double(-1)
    longitude = c_double(-1)
    altitude = c_double(-1)

    status = pcrGetGPSInfo(handle, to_pcrbool(refresh), byref(updated), byref(sec_since_epoch),
                           byref(latitude), byref(longitude), byref(altitude))

    return {
        'status': status,

        'updated': from_pcrbool(updated.value),
        'sec_since_epoch': sec_since_epoch.value,
        'latitude': latitude.value,
        'longitude': longitude.value,
        'altitude': altitude.value,
    }

@error_check
def pcr_write_message_to_gps(handle: int, key: int, value: int, value_size_bytes: int) -> dict[str, any]:
    status = pcrWriteMessageToGPS(handle, key, value, value_size_bytes)

    return {
        'status': status
    }

@error_check
def pcr_set_fan_enabled(handle: int, enabled: bool) -> dict[str, any]:
    status = pcrSetFanEnabled(handle, to_pcrbool(enabled))

    return {
        'status': status
    }

@error_check
def pcr_get_fan_enabled(handle: int) -> dict[str, any]:
    enabled = c_int(-1)

    status = pcrGetFanEnabled(handle, byref(enabled))

    return {
        'status': status,

        'enabled': from_pcrbool(enabled.value)
    }

@error_check
def pcr_set_fan_set_point(handle: int, set_point: float) -> dict[str, any]:
    status = pcrSetFanSetPoint(handle, set_point)

    return {
        'status': status
    }

@error_check
def pcr_get_fan_set_point(handle: int) -> dict[str, any]:
    set_point = c_float(-1)

    status = pcrGetFanSetPoint(handle, byref(set_point))

    return {
        'status': status,

        'set_point': set_point.value
    }

@error_check
def pcr_get_channel_count(handle: int) -> dict[str, any]:
    channel_count = c_int(-1)

    status = pcrGetChannelCount(handle, byref(channel_count))

    return {
        'status': status,

        'channel_count': channel_count.value
    }

@error_check
def pcr_set_channel_config(handle: int, enabled: list[bool], sweep_channel: int) -> dict[str, any]:
    enabled = (c_int * 4)(*[1 if x else 0 for x in enabled])

    status = pcrSetChannelConfig(handle, enabled, sweep_channel)

    return {
        'status': status
    }

@error_check
def pcr_get_channel_config(handle: int) -> dict[str, any]:
    enabled = (c_int * 4)(-1, -1, -1, -1)
    sweep_channel = c_int(-1)

    status = pcrGetChannelConfig(handle, enabled, byref(sweep_channel))

    return {
        'status': status,

        'enabled': [True if x == 1 else False for x in list(enabled)],
        'sweep_channel': sweep_channel.value
    }

@error_check
def pcr_set_channel_shared(handle: int, channel: int, use_shared_lo: bool) -> dict[str, any]:
    status = pcrSetChannelShared(handle, channel, to_pcrbool(use_shared_lo))

    return {
        'status': status
    }

@error_check
def pcr_get_channel_shared(handle: int, channel: int) -> dict[str, any]:
    use_shared_lo = c_int(-1)

    status = pcrGetChannelShared(handle, channel, byref(use_shared_lo))

    return {
        'status': status,

        'use_shared_lo': from_pcrbool(use_shared_lo.value)
    }

@error_check
def pcr_set_channel_freq(handle: int, channel: int, freq_hz: float) -> dict[str, any]:
    status = pcrSetChannelFreq(handle, channel, freq_hz)

    return {
        'status': status
    }

@error_check
def pcr_get_channel_freq(handle: int, channel: int) -> dict[str, any]:
    freq_hz = c_double(-1)

    status = pcrGetChannelFreq(handle, channel, byref(freq_hz))

    return {
        'status': status,

        'freq_hz': freq_hz.value
    }

@error_check
def pcr_set_channel_phase_offset(handle: int, channel: int, phase_offset: float) -> dict[str, any]:
    status = pcrSetChannelPhaseOffset(handle, channel, phase_offset)

    return {
        'status': status
    }

@error_check
def pcr_get_channel_phase_offset(handle: int, channel: int) -> dict[str, any]:
    phase_offset = c_double(-1)

    status = pcrGetChannelPhaseOffset(handle, channel, byref(phase_offset))

    return {
        'status': status,

        'phase_offset': phase_offset.value
    }

@error_check
def pcr_set_channel_delay(handle: int, channel: int, delay: float) -> dict[str, any]:
    status = pcrSetChannelDelay(handle, channel, delay)

    return {
        'status': status
    }

@error_check
def pcr_get_channel_delay(handle: int, channel: int) -> dict[str, any]:
    delay = c_double(-1)

    status = pcrGetChannelDelay(handle, channel, byref(delay))

    return {
        'status': status,

        'delay': delay.value
    }

@error_check
def pcr_set_shared_freq(handle: int, freq_hz: float) -> dict[str, any]:
    status = pcrSetSharedFreq(handle, freq_hz)

    return {
        'status': status
    }

@error_check
def pcr_get_shared_freq(handle: int) -> dict[str, any]:
    freq_hz = c_double(-1)

    status = pcrGetSharedFreq(handle, byref(freq_hz))

    return {
        'status': status,

        'freq_hz': freq_hz.value
    }

@error_check
def pcr_set_stream_data_type(handle: int, data_type: PCRDataType) -> dict[str, any]:
    status = pcrSetStreamDataType(handle, data_type)

    return {
        'status': status
    }

@error_check
def pcr_set_stream_full_scale(handle: int, full_scale: bool) -> dict[str, any]:
    status = pcrSetStreamFullScale(handle, to_pcrbool(full_scale))

    return {
        'status': status
    }

@error_check
def pcr_set_stream_ref_level(handle: int, ref_level: float) -> dict[str, any]:
    status = pcrSetStreamRefLevel(handle, ref_level)

    return {
        'status': status
    }

@error_check
def pcr_set_stream_atten(handle: int, atten: int) -> dict[str, any]:
    status = pcrSetStreamAtten(handle, atten)

    return {
        'status': status
    }

@error_check
def pcr_set_stream_sample_rate(handle: int, decimation: int) -> dict[str, any]:
    status = pcrSetStreamSampleRate(handle, decimation)

    return {
        'status': status
    }

@error_check
def pcr_set_stream_bandwidth(handle: int, bandwidth: float) -> dict[str, any]:
    status = pcrSetStreamBandwidth(handle, bandwidth)

    return {
        'status': status
    }

@error_check
def pcr_set_stream_trigger_edge(handle: int, edge: PCRTriggerEdge) -> dict[str, any]:
    status = pcrSetStreamTriggerEdge(handle, edge)

    return {
        'status': status
    }

@error_check
def pcr_set_stream_mode(handle: int, mode: PCRStreamMode) -> dict[str, any]:
    status = pcrSetStreamMode(handle, mode)

    return {
        'status': status
    }

@error_check
def pcr_set_stream_local_addr(handle: int, port: int) -> dict[str, any]:
    status = pcrSetStreamLocalAddr(handle, port)

    return {
        'status': status
    }

@error_check
def pcr_set_stream_vrt_addr(handle: int, channel: int, dst_addr: str, dst_port: int, dst_mac_addr: int) -> dict[str, any]:
    status = pcrSetStreamVrtAddr(handle, channel, dst_addr, dst_port, dst_mac_addr)

    return {
        'status': status
    }

@error_check
def pcr_set_stream_vrt_channel_id(handle: int, channel: int, stream_id: int) -> dict[str, any]:
    status = pcrSetStreamVrtChannelID(handle, channel, stream_id)

    return {
        'status': status
    }

@error_check
def pcr_set_stream_vrt_cntx_interval(handle: int, enabled: bool, interval: int) -> dict[str, any]:
    status = pcrSetStreamVrtCntxInterval(handle, to_pcrbool(enabled), interval)

    return {
        'status': status
    }

@error_check
def pcr_set_stream_vrt_endianness(handle: int, endianness: PCREndianness) -> dict[str, any]:
    status = pcrSetStreamVrtEndianness(handle, endianness)

    return {
        'status': status
    }

@error_check
def pcr_set_sweep_ref_level(handle: int, ref_level: float) -> dict[str, any]:
    status = pcrSetSweepRefLevel(handle, ref_level)

    return {
        'status': status
    }

@error_check
def pcr_set_sweep_atten(handle: int, atten: int) -> dict[str, any]:
    status = pcrSetSweepAtten(handle, atten)

    return {
        'status': status
    }

@error_check
def pcr_set_sweep_center_span(handle: int, center_hz: float, span_hz: float) -> dict[str, any]:
    status = pcrSetSweepCenterSpan(handle, center_hz, span_hz)

    return {
        'status': status
    }

@error_check
def pcr_set_sweep_start_stop(handle: int, start_hz: float, stop_hz: float) -> dict[str, any]:
    status = pcrSetSweepStartStop(handle, start_hz, stop_hz)

    return {
        'status': status
    }

@error_check
def pcr_set_sweep_coupling(handle: int, rbw: float, vbw: float, sweep_time: float) -> dict[str, any]:
    status = pcrSetSweepCoupling(handle, rbw, vbw, sweep_time)

    return {
        'status': status
    }

@error_check
def pcr_set_sweep_detector(handle: int, detector: PCRSweepDetector, video_units: PCRSweepVideoUnits) -> dict[str, any]:
    status = pcrSetSweepDetector(handle, detector, video_units)

    return {
        'status': status
    }

@error_check
def pcr_set_sweep_scale(handle: int, scale: PCRSweepScale) -> dict[str, any]:
    status = pcrSetSweepScale(handle, scale)

    return {
        'status': status
    }

@error_check
def pcr_set_sweep_window(handle: int, window: PCRSweepWindow) -> dict[str, any]:
    status = pcrSetSweepWindow(handle, window)

    return {
        'status': status
    }

@error_check
def pcr_set_vsg_enabled(handle: int, enabled: bool) -> dict[str, any]:
    status = pcrSetVSGEnabled(handle, to_pcrbool(enabled))

    return {
        'status': status
    }

@error_check
def pcr_set_vsg_level(handle: int, level: float) -> dict[str, any]:
    status = pcrSetVSGLevel(handle, level)

    return {
        'status': status
    }

@error_check
def pcr_set_vsg_pattern(handle: int, iq: np.typing.NDArray, length: int, data_type: PCRDataType) -> dict[str, any]:
    iq_c = (iq.__array_interface__['data'][0]
            + np.arange(iq.shape[0]) * iq.strides[0]).astype(np.uintp)

    status = pcrSetVSGPattern(handle, iq_c, length, data_type)

    return {
        'status': status
    }

@error_check
def pcr_initiate(handle: int) -> dict[str, any]:
    status = pcrInitiate(handle)

    return {
        'status': status
    }

@error_check
def pcr_abort(handle: int) -> dict[str, any]:
    status = pcrAbort(handle)

    return {
        'status': status
    }

@error_check
def pcr_stream_parameters(handle: int) -> dict[str, any]:
    channel_count = c_int(-1)
    sample_rate = c_double(-1)
    bandwidth = c_double(-1)

    status = pcrStreamParameters(handle, byref(channel_count), byref(sample_rate), byref(bandwidth))

    return {
        'status': status,

        'channel_count': channel_count.value,
        'sample_rate': sample_rate.value,
        'bandwidth': bandwidth.value
    }

@error_check
def pcr_stream_correction(handle: int, channel: int, correction: float) -> dict[str, any]:
    status = pcrStreamCorrection(handle, channel, correction)

    return {
        'status': status
    }

@error_check
def pcr_stream_recv(handle: int, buffers: np.typing.NDArray, samples: int,
                    purge: bool, triggers: np.typing.NDArray[np.float64] = None,
                    trigger_buf_size: int = 0) -> dict[str, any]:
    ns_since_epoch = c_int64(-1)
    sample_loss = c_int(-1)
    samples_remaining = c_int(-1)

    buffers_c = (buffers.__array_interface__['data'][0]
                 + np.arange(buffers.shape[0]) * buffers.strides[0]).astype(np.uintp)

    triggers_c = triggers.ctypes.data_as(POINTER(c_double)) if triggers is not None else None

    status = pcrStreamRecv(handle, buffers_c, samples, byref(ns_since_epoch), to_pcrbool(purge),
             byref(sample_loss), byref(samples_remaining), triggers_c, trigger_buf_size)

    return {
        'status': status,

        'ns_since_epoch': ns_since_epoch.value,
        'sample_loss': sample_loss.value,
        'samples_remaining': samples_remaining.value
    }

@error_check
def pcr_sweep_get_parameters(handle: int) -> dict[str, any]:
    actual_rbw = c_double(-1)
    actual_vbw = c_double(-1)
    actual_start_freq = c_double(-1)
    bin_size = c_double(-1)
    sweep_size = c_int(-1)

    status = pcrSweepGetParameters(handle, byref(actual_rbw), byref(actual_vbw),
                                   byref(actual_start_freq), byref(bin_size), byref(sweep_size))

    return {
        'status': status,

        'actual_rbw': actual_rbw.value,
        'actual_vbw': actual_vbw.value,
        'actual_start_freq': actual_start_freq.value,
        'bin_size': bin_size.value,
        'sweep_size': sweep_size.value,
    }

@error_check
def pcr_get_sweep(handle: int, sweep: np.ndarray[np.float32]) -> dict[str, any]:
    status = pcrGetSweep(handle, sweep)

    return {
        'status': status
    }

@error_check
def pcr_sweep_start(handle: int) -> dict[str, any]:
    status = pcrSweepStart(handle)

    return {
        'status': status
    }

@error_check
def pcr_sweep_finish(handle: int, sweep: np.ndarray[np.float32]) -> dict[str, any]:
    status = pcrSweepFinish(handle, sweep)

    return {
        'status': status
    }

@error_check
def pcr_network_config_get_device_list() -> dict[str, any]:
    serials = (c_int * PCR_MAX_DEVICES)()
    device_count = c_int(PCR_MAX_DEVICES)

    status = pcrNetworkConfigGetDeviceList(serials, byref(device_count))

    return {
        'status': status,

        'serials': list(serials),
        'device_count': device_count.value
    }

@error_check
def pcr_network_config_open_device(serial_number: int) -> dict[str, any]:
    handle = c_int(-1)

    status = pcrNetworkConfigOpenDevice(byref(handle), serial_number)

    return {
        'status': status,

        'handle': handle.value
    }

@error_check
def pcr_network_config_close_device(handle: int) -> dict[str, any]:
    status = pcrNetworkConfigCloseDevice(handle)

    return {
        'status': status
    }

@error_check
def pcr_network_config_get_mac(handle: int) -> dict[str, any]:
    mac = c_char_p(b'XX-XX-XX-XX-XX-XX')

    status = pcrNetworkConfigGetMAC(handle, mac)

    return {
        'status': status,

        'mac': mac.value.decode()
    }

@error_check
def pcr_network_config_set_ip(handle: int, addr: str, non_volatile: bool) -> dict[str, any]:
    status = pcrNetworkConfigSetIP(handle, addr.encode(), to_pcrbool(non_volatile))

    return {
        'status': status
    }

@error_check
def pcr_network_config_get_ip(handle: int) -> dict[str, any]:
    ip = c_char_p(b'xxx.xxx.xxx.xxx')

    status = pcrNetworkConfigGetIP(handle, ip)

    return {
        'status': status,

        'ip': ip.value.decode()
    }

@error_check
def pcr_network_config_set_port(handle: int, port: int, non_volatile: bool) -> dict[str, any]:
    status = pcrNetworkConfigSetPort(handle, port, to_pcrbool(non_volatile))

    return {
        'status': status
    }

@error_check
def pcr_network_config_get_port(handle: int) -> dict[str, any]:
    port = c_int(-1)

    status = pcrNetworkConfigGetPort(handle, byref(port))

    return {
        'status': status,

        'port': port.value
    }

def pcr_get_api_version() -> str:
    return pcrGetAPIVersion().decode('ascii')

def pcr_get_error_string(status: PCRStatus) -> str:
    return pcrGetErrorString(status).decode()
