#include "pcr_api.h"

#include <cassert>
#include <iostream>
#include <vector>

// This example opens the PCR4200 and configures the device for a basic
// sweep on one channel. Once configured, a sweep is performed.

static void checkStatus(PCRStatus sts)
{
    if(sts < PCRStatusNoError) {
        printf("Status error: %s\n", pcrGetErrorString(sts));
        assert(false);
    } else if(sts > PCRStatusNoError) {
        printf("Status warning: %s\n", pcrGetErrorString(sts));
    }
}

void pcr_example_sweep_basic()
{
    PCRStatus sts;
    int handle;

    // Connect the device
    sts = pcrConnectDevice(&handle, PCR_DEFAULT_HOST_ADDR, PCR_DEFAULT_DEVICE_ADDR, PCR_DEFAULT_PORT);
    if(sts != PCRStatusNoError) {
        printf("Error opening device: %s\n", pcrGetErrorString(sts));
        return;
    }

    // Configure reference level to -20dBm
    // The input signal should be <= -20dBm
    sts = pcrSetSweepAtten(handle, PCR_AUTO_ATTEN);
    checkStatus(sts);
    sts = pcrSetSweepRefLevel(handle, -20.0);
    checkStatus(sts);

    // Enable one sweep channel, between [0,3]
    const int activeChannel = 0;
    int chEnabled[4] = { 0, 0, 0, 0 };
    chEnabled[activeChannel] = 1;

    // These function configure a sweep on the specified channel, and 
    //   no streaming channels. Sweep channels cannot used the shared LO, but
    //   we disable it here anyways for clarity.
    sts = pcrSetChannelShared(handle, activeChannel, PCRBoolFalse);
    checkStatus(sts);
    sts = pcrSetChannelConfig(handle, chEnabled, activeChannel);
    checkStatus(sts);

    // Configure sweep sweep configuration
    sts = pcrSetSweepStartStop(handle, PCR_MIN_FREQ, PCR_MAX_FREQ);
    checkStatus(sts);
    sts = pcrSetSweepCoupling(handle, 100.0e3, 100.0e3, 0.001);
    checkStatus(sts);
    sts = pcrSetSweepDetector(handle, PCRSweepDetectorMax, PCRSweepVideoUnitsPower);
    checkStatus(sts);
    sts = pcrSetSweepScale(handle, PCRSweepScaleLog);
    checkStatus(sts);
    sts = pcrSetSweepWindow(handle, PCRSweepWindowFlattop);
    checkStatus(sts);

    // Start the measurements
    sts = pcrInitiate(handle);
    checkStatus(sts);

    // Query the resulting sweep parameters
    double actualRbw, actualVbw, actualStart, binSize;
    int sweepSize;
    sts = pcrSweepGetParameters(handle, &actualRbw, &actualVbw, &actualStart, &binSize, &sweepSize);
    checkStatus(sts);

    // Allocate memory for the sweep
    std::vector<float> sweep(sweepSize);

    // Do a sweep
    sts = pcrGetSweep(handle, &sweep[0]);
    checkStatus(sts);

    // The sweep is now stored in the sweep vector. The frequency of any point
    // in the sweep vector is calculated as,
    // freq = actualStart + binSize * index

    // Do something with the sweep data here...

    // Done, close device
    pcrCloseDevice(handle);
}