#include "pcr_api.h"

#include <cassert>
#include <cstdlib>
#include <cstdio>
#include <vector>

#include <iostream>
#include <thread>

#include <windows.h>

// This example illustrates how fast retuning might be implemented in a system
//   that is sweeping and streaming simultaneously. 
// The PCR is configured in the main thread to sweep on channel 1, 
//   and stream on channels 2,3,4.
// Once the measurements are initialized, two threads are started, one for sweeping
//   and one for streaming.
// The main thread then issues a series of retune events while the other threads
//   are actively measuring. 
// The measurements are considered complete after a period, the measurement threads
//   are ended, and the device is closed.
// Timestamps are collected for each retune event.

static void sweep_thread(int handle, bool *running)
{
    double actualRBW, actualVBW, actualStart, actualBin;
    int sweepSize;
    PCRStatus sts = pcrSweepGetParameters(handle, &actualRBW, &actualVBW, &actualStart, &actualBin, &sweepSize);
    assert(sts == PCRStatusNoError);

    std::vector<float> sweep(sweepSize);

    int counter = 0;

    int start = GetTickCount();

    while(*running) {
        sts = pcrGetSweep(handle, &sweep[0]);
        assert(sts == PCRStatusNoError);
        printf("Sweep %d\n", ++counter);
    }

    printf("Elapsed %d\n", GetTickCount() - start);

    int bp = 0;
}

static void stream_thread(int handle, bool *running)
{
    int channelCount;
    double sampleRate, bandwidth;
    PCRStatus sts = pcrStreamParameters(handle, &channelCount, &sampleRate, &bandwidth);

    // Request 100ms of data on each call
    int samplesPerAcq = sampleRate / 10;

    // Setup our buffers/pointers
    void *bufPts[4] = { 0 };
    std::vector<float> bufs[4];
    for(int i = 0; i < channelCount; i++) {
        bufs[i].resize(samplesPerAcq * 2);
        bufPts[i] = &bufs[i][0];
    }

    int count = 0;
    while(*running) {
        sts = pcrStreamRecv(handle, bufPts, samplesPerAcq, 0, 
            PCRBoolFalse, 0, 0, 0, 0);

        printf("Streaming here %d\n", count++);
    }
}

void pcr_example_fast_retune()
{
    PCRStatus sts;
    int handle;
    sts = pcrConnectDevice(&handle, PCR_DEFAULT_HOST_ADDR, PCR_DEFAULT_DEVICE_ADDR, PCR_DEFAULT_PORT);
    assert(sts == PCRStatusNoError);

    const int CHANNELS = 2;

    int chEnabled[4] = { 1, 1, 1, 1 };
    sts = pcrSetChannelConfig(handle, chEnabled, 0);
    assert(sts == PCRStatusNoError);
    for(int i = 0; i < 4; i++) {
        sts = pcrSetChannelShared(handle, i, PCRBoolFalse);
        assert(sts == PCRStatusNoError);
        sts = pcrSetChannelFreq(handle, i, 1.5e9);
        assert(sts == PCRStatusNoError);
    }

    // Configure sweep
    sts = pcrSetSweepRefLevel(handle, -20.0);
    assert(sts == PCRStatusNoError);
    sts = pcrSetSweepAtten(handle, PCR_AUTO_ATTEN);
    assert(sts == PCRStatusNoError);
    sts = pcrSetSweepStartStop(handle, 100.0e3, 20.0e9);
    assert(sts == PCRStatusNoError);
    sts = pcrSetSweepCoupling(handle, 100.0e3, 100.0e3, 1.0e-6);
    assert(sts == PCRStatusNoError);
    sts = pcrSetSweepDetector(handle, PCRSweepDetectorMax, PCRSweepVideoUnitsPower);
    assert(sts == PCRStatusNoError);
    sts = pcrSetSweepScale(handle, PCRSweepScaleLog);
    assert(sts == PCRStatusNoError);
    sts = pcrSetSweepWindow(handle, PCRSweepWindowFlattop);
    assert(sts == PCRStatusNoError);

    // Configure streams
    sts = pcrSetSharedFreq(handle, 1.0e9);
    assert(sts == PCRStatusNoError);
    sts = pcrSetStreamDataType(handle, PCRDataType32fc);
    assert(sts == PCRStatusNoError);
    sts = pcrSetStreamRefLevel(handle, -20.0);
    assert(sts == PCRStatusNoError);
    sts = pcrSetStreamAtten(handle, PCR_AUTO_ATTEN);
    assert(sts == PCRStatusNoError);
    sts = pcrSetStreamSampleRate(handle, 1);
    assert(sts == PCRStatusNoError);
    sts = pcrSetStreamBandwidth(handle, PCR_STREAM_MAX_BW);
    assert(sts == PCRStatusNoError);

    sts = pcrInitiate(handle);
    assert(sts == PCRStatusNoError);

    std::cout << "Device is configured\n";

    bool running = true;
    std::thread sweepThreadHandle(&sweep_thread, handle, &running);
    std::thread streamThreadHandle(&stream_thread, handle, &running);

    int64_t timeStamp;
    std::vector<int64_t> times;

    // How many retunes we do
    const int retuneCount = 1000;
    // How much time in ms to wait between each retune, range [0, max int]
    const int retuneDelayMs = 0;

    srand(GetTickCount());

    for(int i = 0; i < retuneCount; i++) {
        // Pick a channel to tune, channels 1,2,3 (0-based) are streaming
        int chToChange = rand() % 3 + 1;
        // Set freq between [1, 10GHz] in 1GHz increments
        double freq = (rand() % 10) * 1.0e9 + 1.0e9;

        sts = pcrUpdateChannelFreq(handle, chToChange, freq, &timeStamp);
        times.push_back(timeStamp);

        Sleep(retuneDelayMs);
    }

    running = false;
    if(sweepThreadHandle.joinable()) {
        sweepThreadHandle.join();
    }
    if(streamThreadHandle.joinable()) {
        streamThreadHandle.join();
    }

    sts = pcrAbort(handle);
    pcrCloseDevice(handle);

}
