# Sceptre LiveStream

Python 3 module for receiving Sceptre VITA-49 LiveStreams.  The primary goal of this project is to provide a reference for customers to integrate with the Sceptre LiveStream plugin.  As a result, code clarity was prioritized over performance.  However, this library should also be internally useful for testing and prototyping.

## Installation:

* User level:
```bash
cd <path-to-sceptre-livestream-repo>
pip3 install --user .
```
* Virtual environment example (can also be added to `requirements.txt` of super project):
```bash
python3 -m venv ./venv
source venv/bin/activate
pip install <path-to-sceptre-livestream-repo>
```

Alternatively, package this Python repo as a wheel, and bring it anywhere (off-network installation requires that the dependencies have been installed already):

1. Use the [new pip installation method](https://stackoverflow.com/a/60786507)
```bash
sudo apt install python3-pip
```

2. Install necessary Python packages for packaging as a wheel file: `python3 -m pip install setuptools wheel`

3. Package this repo as a wheel: `python3 setup.py bdist_wheel`

4. The wheel file will appear in the generated `dist/` folder.

5. Install the wheel where ever necessary: `python3 -m pip install ./sceptrelivestream-1.0.0-py3-none-any.whl`

## Usage

Let's start with a simple example of how to receive a Sceptre LiveStream:

```python
from sceptrelivestream import LiveStream, IFFrame, SpectralFrame

# Create livestream object
livestream = LiveStream('', 12000)

while True:
    # Read the next VRT packet
    pkt = livestream.read()
    
    if isinstance(pkt.frame, IFFrame):
        # This is an IF data frame.
        ...
    elif isinstance(pkt.frame, SpectralFrame):
        # This is a spectral data frame
        ...
```

The `LiveStream` object is the primary top-level interface for reading a Sceptre LiveStream from the network.  `LiveStream.read()` reads a single UDP packet from the network and attempts to decode the payload as a VRT (VITA Radio Transport) packet.  If a VRT packet is successfully decoded, a `VRTPacket` object is returned:

```python
class VRTPacket:
    header: VRTHeader
    context: VRTContext | None
    frame: IFFrame | SpectralFrame | None
```

The VRT header and associated context packet are provided in `VRTPacket` to provide lower-level metadata about the contents of the decoded packet, but the `frame` member is the primary way to access the data payload of the packet.  Not all `VRTPacket`s contain a data frame, so it is important to check that it exists before accessing it.

`sceptre-livestream` supports time-domain IF data and frequency-domain spectral data.  `VRTPacket.frame` can either be an `IFFrame` or a `SpectralFrame`.  Each frame type contains its own set of metadata:

```python
class IFFrame:
    timestamp: Timestamp
    sample_delta: float
    center_frequency: float
    gain_dB: float
    bandwidth: float
    data: np.ndarray

class SpectralFrame:
    timestamp: Timestamp
    frame_delta: float
    frequency_start: float
    frequency_delta: float
    data: np.ndarray
```

Data is returned in NumPy arrays.  By default, data is converted to floating point (32-bit, real or complex) and scaled based on the gain values in the VRT context packets.  The raw data format can be returned if the `raw_output` constructor argument of `LiveStream` is set to true, however NumPy does not support complex integer formats natively, so the raw format may be more difficult to work with.

The size of a VRT packet is usually chosen based on network MTU (often 1500 bytes for 1GbE and 9000 bytes for 10GbE).  The maximum UDP packet size is 64KB (including headers), so that is the cap for Sceptre LiveStreams.  Often spectral frames are larger than the VRT packet size, so Sceptre uses a custom extension packet format to send fragments of spectral frames in each VRT packet.  These fragments are re-assembled by `sceptre-livestream` to form `SpectralFrame`s.  Because of this, when streaming spectral data, you may receive many `VRTPacket` objects from `LiveStream.read()` before receiving a packet that contains a `SpectralFrame`.  

## Extras

### `PSDEngine`
`sceptre-livestream` includes a `PSDEngine` object that computes the power spectral density from time-domain data.  `PSDEngine` rechunks the VRT IF packets into frames of the FFT size, performs the FFT, converts to mag^2, averages mag^2 frames, and finally converts to dB.  This follows the same process Sceptre uses to produce spectral data. 

### Plot Spectrum
The `sceptre-livestream` repo contains a demo script (`plot_spectrum.py`) that streams in either IF or spectral data and plots it using `matplotlib`.  If the stream contains IF data, `PSDEngine` is used to compute spectral data.

## Future Work
One major aspect of Sceptre LiveStreams this library does not yet address is VITA-49.2 command and control.  I would like to add functions like `LiveStream.set_frequency()`, etc. to allow the user to control the LiveStream in the ways provided by VITA-49.2 messages.