﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using Ivi.Visa;

// This example demonstrates 
// 1. Use VISA to connect to Spike 
// 2. Put the device into non-continuous mode 
// 4. Perform and wait for a sweep to complete
// 5. Obtain trace info 
// 6. Obtain trace data

namespace scpi_csharp_example
{
    class Program
    {
        static void Main(string[] args)
        {
            Ivi.Visa.Interop.ResourceManager rm;
            Ivi.Visa.Interop.FormattedIO488 inst;
            int opc;

            // Get the VISA resource manager and message-based session
            rm = new Ivi.Visa.Interop.ResourceManager();
            inst = new Ivi.Visa.Interop.FormattedIO488();

            // Open a session to the Spike software, Spike must be running at this point
            inst.IO = (Ivi.Visa.Interop.IMessage)rm.Open("TCPIP::localhost::5025::SOCKET");

            // For SOCKET programming, we want to tell VISA to use a terminating character 
            //   to end a read operation. In this case we want the newline character to end a 
            //   read operation. The termchar is typically set to newline by default. Here we
            //   set it for illustrative purposes.
            inst.IO.TerminationCharacterEnabled = true;
            inst.IO.TerminationCharacter = (byte)'\n';

            // Set the measurement mode to sweep
            inst.WriteString("INSTRUMENT:SELECT SA");
            // Disable continuous meausurement operation
            inst.WriteString("INIT:CONT OFF");

            // Configure a 20MHz span sweep at 1GHz
            // Set the RBW/VBW to auto
            inst.WriteString("SENS:BAND:RES:AUTO ON; :BAND:VID:AUTO ON; :BAND:SHAPE FLATTOP");
            // Center/span
            inst.WriteString("SENS:FREQ:SPAN 20MHZ; CENT 1GHZ");
            // Reference level/Div
            inst.WriteString("SENS:POW:RF:RLEV -20DBM; PDIV 10");
            // Peak detector
            inst.WriteString("SENS:SWE:DET:FUNC MINMAX; UNIT POWER");

            // Configure the trace. Ensures trace 1 is active and enabled for clear-and-write.
            // These commands are not required to be sent everytime, this is for illustrative purposes only. 
            inst.WriteString("TRAC:SEL 1"); // Select trace 1
            inst.WriteString("TRAC:TYPE WRITE"); // Set clear and write mode            
            inst.WriteString("TRAC:UPD ON"); // Set update state to on
            inst.WriteString("TRAC:DISP ON"); // Set un-hidden

            // Trigger a sweep, and wait for it to complete
            inst.WriteString(":INIT; *OPC?");            
            opc = (int)inst.ReadNumber();
            if (opc != 1) 
            {
                // Sweep not completed
                return;
            }

            // Get trace info            
            inst.WriteString("SENS:FREQ:STAR?"); // X-start (start frequency)
            double xStart = inst.ReadNumber();

            inst.WriteString("TRAC:XINC?"); // X-increment (bin size)
            double xIncrement = inst.ReadNumber();

            inst.WriteString("TRAC:POINTS?"); // Sweep length
            int sweepLen = (int)inst.ReadNumber();

            // Get trace data
            inst.WriteString(":TRAC:DATA?");
            double[] results = inst.ReadList(Ivi.Visa.Interop.IEEEASCIIType.ASCIIType_R8, ",");
                
            // For each point (i) in sweepLen:
            //     frequency (Hz) = xStart + i * xIncrement
            //     amplitude (dBm) = results[i]

            // Done
            inst.IO.Close();            
        }
    }
}
