# -*- coding: utf-8 -*-

# Copyright (c) 2025 Signal Hound
# For licensing information, please see the API license in the software_licenses folder

from ctypes import CDLL
from ctypes import byref
from ctypes import c_int
from ctypes import c_uint

from enum import Enum

from sys import exit

rfslib = CDLL("rfsdevice/rfs_api.so")


# ---------------------------------- Constants ---------------------------------

RFS_INVALID_HANDLE = -1

MAX_RF_PORTS = 8

# Status code returned from all RFS API functions
class RfsStatus(Enum):
    # Invalid device handle
    rfsInvalidDeviceHandle = -6
    # Invalid device info
    rfsInvalidDeviceInfo = -5
    # Invalid RF port number
    rfsInvalidPortNumberErr = -4
    # Invalid serial port number
    rfsInvalidSerialPortNumberErr = -3
    # Device disconnected
    rfsConnectionLostErr = -2
    # Unable to open device
    rfsDeviceNotFoundErr = -1

    # Function returned successfully
    rfsNoError = 0

# Device type
class RfsDeviceType(Enum):
    # RFS8
    RfsDeviceTypeRFS8 = 0
    # RFS44
    RfsDeviceTypeRFS44 = 1


# --------------------------------- Bindings -----------------------------------

rfsOpenDevice = rfslib.rfsOpenDevice
rfsCloseDevice = rfslib.rfsCloseDevice

rfsSetPort = rfslib.rfsSetPort
rfsGetPort = rfslib.rfsGetPort

rfsQuickSetPort = rfslib.rfsQuickSetPort
rfsGetDeviceInfo = rfslib.rfsGetDeviceInfo


# ---------------------------------- Utility ----------------------------------

def error_check(func):
    def print_status_if_error(*args, **kwargs):
        return_vars = func(*args, **kwargs)
        if "status" not in return_vars.keys():
            return return_vars
        status = return_vars["status"]
        if status != 0:
            print (f"{'Error' if status < 0 else 'Warning'} {status}: Something went wrong in {func.__name__}()")
        if status < 0:
            exit()
        return return_vars
    return print_status_if_error


# --------------------------------- Functions ---------------------------------

@error_check
def rfs_open_device(serial_port):
    device = c_int(-1)

    status = rfsOpenDevice(serial_port, byref(device))

    return {
        "status": status,

        "device": device.value
    }

@error_check
def rfs_close_device(device):
    return {
        "status": rfsCloseDevice(device)
    }

@error_check
def rfs_set_port(device, port):
    return {
        "status": rfsSetPort(device, port)
    }

@error_check
def rfs_get_port(device):
    port = c_int(-1)

    status = rfsGetPort(device, byref(port))

    return {
        "status": status,

        "port": port.value
    }

@error_check
def rfs_quick_set_port(serial_port, port):
    return {
        "status": rfsQuickSetPort(serial_port, port)
    }

@error_check
def rfs_get_device_info(device):
    device_type = c_int(-1)
    sid = c_uint(0)
    firmware = c_uint(0)

    status = rfsGetDeviceInfo(device, byref(device_type), byref(sid), byref(firmware))

    return {
        "status": status,

        "device_type": RfsDeviceType(device_type.value),
        "sid": sid.value,
        "firmware": firmware.value
    }
