# This example opens the PCR4200 and configures the device for a basic
# sweep on one channel. Once configured, a sweep is performed.

from pcrdevice.pcr_api import *

import matplotlib.pyplot as plt

def pcr_example_sweep_basic():
    # Connect the device
    status, handle = pcr_connect_device(PCR_DEFAULT_HOST_ADDR, PCR_DEFAULT_DEVICE_ADDR, PCR_DEFAULT_PORT).values()
    if status != PCRStatus.PCRStatusNoError:
        print(f'Error opening device: {pcr_get_error_string(status)}')
        return

    # Configure reference level to -20dBm
    # The input signal should be <= -20dBm
    pcr_set_sweep_atten(handle, PCR_AUTO_ATTEN)
    pcr_set_sweep_ref_level(handle, -20.0)

    # Enable one sweep channel, between [0,3]
    ACTIVE_CHANNEL = 0
    ch_enabled = [False, False, False, False]
    ch_enabled[ACTIVE_CHANNEL] = True

    # These function configure a sweep on the specified channel, and
    #   no streaming channels. Sweep channels cannot used the shared LO, but
    #   we disable it here anyways for clarity.
    pcr_set_channel_shared(handle, ACTIVE_CHANNEL, False)
    pcr_set_channel_config(handle, ch_enabled, ACTIVE_CHANNEL)

    # Configure sweep sweep configuration
    pcr_set_sweep_start_stop(handle, PCR_MIN_FREQ, PCR_MAX_FREQ)
    pcr_set_sweep_coupling(handle, 100.0e3, 100.0e3, 0.001)
    pcr_set_sweep_detector(handle, PCRSweepDetector.PCRSweepDetectorMax, PCRSweepVideoUnits.PCRSweepVideoUnitsPower)
    pcr_set_sweep_scale(handle, PCRSweepScale.PCRSweepScaleLog)
    pcr_set_sweep_window(handle, PCRSweepWindow.PCRSweepWindowFlattop)

    # Start the measurements
    pcr_initiate(handle)

    # Query the resulting sweep parameters
    status, actual_rbw, actual_vbw, actual_start, bin_size, sweep_size = pcr_sweep_get_parameters(handle).values()

    # # Allocate memory for the sweep
    sweep = np.zeros(sweep_size).astype(np.float32)

    # Do a sweep
    pcr_get_sweep(handle, sweep)

    # The sweep is now stored in the sweep array. The frequency of any point
    # in the sweep array is calculated as,
    # freq = actualStart + binSize * index

    # Do something with the sweep data here...
    print(f'Peak: {sweep.max():.2f} dBm')

    # Plot sweep data
    freqs = [actual_start + i * bin_size for i in range(sweep_size)]
    plt.plot(freqs, sweep)
    plt.show()

    # Done, close device
    pcr_close_device(handle)


if __name__ == '__main__':
    pcr_example_sweep_basic()
