#include "pcr_api.h"

#include <iostream>
#include <cassert>
#include <vector>
#include <complex>

// This example configures the onboard signal generator for a CW output
// and configures a single channel I/Q reciever. 

static void checkStatus(PCRStatus sts)
{
    if(sts < PCRStatusNoError) {
        printf("Status error: %s\n", pcrGetErrorString(sts));
        assert(false);
    } else if(sts > PCRStatusNoError) {
        printf("Status warning: %s\n", pcrGetErrorString(sts));
    }
}

static double avgPowerDBM(std::complex<float>* src, int len)
{
    double p = 0.0;
    for(int i = 0; i < len; i++) {
        p += src[i].real() * src[i].real() + src[i].imag() * src[i].imag();
    }

    return 10.0 * log10(p / len);
}

void pcr_example_vsg_and_rx()
{
    PCRStatus sts;
    int handle;

    // Decimation should be a power of 2
    const int decimation = 16;
    const double expectedSampleRate = PCR_STREAM_SR / decimation;
    float frequency = 2.0e9;

    // Connect the device
    sts = pcrConnectDevice(&handle, PCR_DEFAULT_HOST_ADDR, PCR_DEFAULT_DEVICE_ADDR, PCR_DEFAULT_PORT);
    if(sts != PCRStatusNoError) {
        printf("Error opening device: %s\n", pcrGetErrorString(sts));
        return;
    }

    // Enable one stream channel, between [0,3]
    const int activeChannel = 0;
    int chEnabled[4] = { 0, 0, 0, 0 };
    chEnabled[activeChannel] = 1;

    // Set channel configuration
    sts = pcrSetChannelConfig(handle, chEnabled, PCR_SWEEP_CHANNEL_DISABLED);
    checkStatus(sts);
    sts = pcrSetChannelShared(handle, activeChannel, PCRBoolTrue);
    checkStatus(sts);
    // VSG uses the shared frequency. So this function will not only control
    //  any receive channels using the shared frequency, but also the VSG output frequency.
    // For this example, the recieve channel will use the shared frequency.
    sts = pcrSetSharedFreq(handle, frequency);
    checkStatus(sts);

    // Configure I/Q stream
    sts = pcrSetStreamDataType(handle, PCRDataType32fc);
    checkStatus(sts);
    sts = pcrSetStreamRefLevel(handle, -20.0);
    checkStatus(sts);
    sts = pcrSetStreamAtten(handle, PCR_AUTO_ATTEN);
    checkStatus(sts);
    sts = pcrSetStreamSampleRate(handle, decimation);
    checkStatus(sts);
    sts = pcrSetStreamBandwidth(handle, 40.0e6);
    checkStatus(sts);
    // We are using the API to retrieve the I/Q samples, and not Vita49 streaming
    sts = pcrSetStreamMode(handle, PCRStreamModeLocal);
    checkStatus(sts);

    // Pattern is provided as interleaved I/Q samples
    // The pattern must be a multiple of 16 I/Q samples
    const int vsgPatternLen = 16;
    std::vector<std::complex<float>> vsgPattern(vsgPatternLen);
    for(int i = 0; i < vsgPatternLen; i++) {
        vsgPattern[i].real(1.0);
        vsgPattern[i].imag(0.0);
    }

    // Configure the VSG
    pcrSetVSGPattern(handle, vsgPattern.data(), vsgPatternLen, PCRDataType32fc);
    checkStatus(sts);
    pcrSetVSGLevel(handle, -25.0);
    checkStatus(sts);
    pcrSetVSGEnabled(handle, PCRBoolTrue);
    checkStatus(sts);

    // Start the measurements
    sts = pcrInitiate(handle);
    checkStatus(sts);

    // Query the stream parameters
    int channelCount;
    double sampleRate, bandwidth;
    sts = pcrStreamParameters(handle, &channelCount, &sampleRate, &bandwidth);
    checkStatus(sts);

    assert(channelCount == 1);
    assert(sampleRate == expectedSampleRate);

    // Number of samples to query at a time
    int N = 1e6;
    std::vector<std::complex<float>> iq(N);

    // Collect 1 second worth of data
    for(int i = 0; i < 50; i++) {
        std::complex<float> *ptr = iq.data();
        sts = pcrStreamRecv(handle, (void**)&ptr, N, 0, PCRBoolFalse, 0, 0, 0, 0);
        checkStatus(sts);

        // Do something with the data here..
        // For this example we will print off average power
        double power = avgPowerDBM(iq.data(), iq.size());
        printf("Power %lf\n", power);
    }

    // Done, close device
    pcrCloseDevice(handle);
}