#include "pcr_api.h"
#include <iostream>
#include <cassert>
#include <thread>
#include <vector>

// This example shows how to configure the PCR to both sweep and stream I/Q
// data simultaneously. A sweeping channel and 3 streaming channels are configured.
// Sweep and I/Q data is then requested in 2 parallel threads.

static void checkStatus(PCRStatus sts)
{
	if (sts < PCRStatusNoError) {
		printf("Status error: %s\n", pcrGetErrorString(sts));
		assert(false);
	}
	else if (sts > PCRStatusNoError) {
		printf("Status warning: %s\n", pcrGetErrorString(sts));
	}
}

void collectSweepData(int handle)
{
	std::cout << "Collecting Sweep Data\n" << std::endl;

	PCRStatus sts;
	// Query the resulting sweep parameters
	double actualRbw, actualVbw, actualStart, binSize;
	int sweepSize;

	sts = pcrSweepGetParameters(handle, &actualRbw, &actualVbw, &actualStart, &binSize, &sweepSize);
	checkStatus(sts);
	// Allocate memory for the sweep
	std::vector<float> sweep(sweepSize);

	while (1) {
		sts = pcrGetSweep(handle, &sweep[0]);
		checkStatus(sts);

		// do something with sweep data
	}
}

void collectStreamData(int handle)
{
	std::cout << "Collecting Stream Data\n" << std::endl;

	PCRStatus sts;
	const int STREAM_CHANNELS = 3; 
	int len = 10.0e6;

	int16_t* bufs[STREAM_CHANNELS];
	float* bufsPwr[STREAM_CHANNELS];
	float corrections[STREAM_CHANNELS];
	for (int i = 0; i < STREAM_CHANNELS; i++) {
		bufs[i] = new int16_t[len * 2]; // len * 2 because each IQ sample is constructed of 2 int16_t values
		bufsPwr[i] = new float[len];
	}

	// obtain corrections for each channel so we can properly scale the data
	for (int i = 0; i < STREAM_CHANNELS; i++) {
		// channel is offset by 1 since we are streaming on channels 1-3 and sweeping on 0
		sts = pcrStreamCorrection(handle, i+1, &corrections[i]); 
	}
	
	while (1) {
		sts = pcrStreamRecv(handle, (void**)bufs, len, 0, PCRBoolTrue, 0, 0, 0, 0);
		assert(sts == PCRStatusNoError);

		/*
		for (int i = 0; i < STREAM_CHANNELS; i++) {
			for (int j = 0; j < len; j++) {
				// Scale 16bit complex ints to floats from -1.0 to 1.0
				float re32f = ((float)bufs[i][j * 2]) / 32768.0;
				float im32f = ((float)bufs[i][j * 2 + 1]) / 32768.0;

				// Apply amplitude corrections
				re32f *= corrections[i];
				im32f *= corrections[i];

				// Calculate Power
				bufsPwr[i][j] = 10.0 * log10(re32f * re32f + im32f * im32f);
			}
		}*/

		// do something with stream data
	}
}

void pcr_example_sweep_and_stream()
{
	PCRStatus sts;
	int handle;
	const int CHANNELS = 4;
	sts = pcrConnectDevice(&handle, PCR_DEFAULT_HOST_ADDR, PCR_DEFAULT_DEVICE_ADDR, PCR_DEFAULT_PORT);
	checkStatus(sts);

	int chEnabled[4] = { 1, 1, 1, 1 };
	int sweepChan = 0;
	sts = pcrSetChannelConfig(handle, chEnabled, sweepChan);
	checkStatus(sts);

	// Configure Sweep
	double rbw = 100.0e3;
	double vbw = 100.0e3;
	double sweepTime = 1.0e-6;
	double sweepRefLevel = -20.0;
	double centerFreq = 10.0e9;
	double span = 20.0e9;
	sts = pcrSetChannelShared(handle, sweepChan, PCRBoolFalse);
	checkStatus(sts);
	sts = pcrSetSweepRefLevel(handle, sweepRefLevel);
	checkStatus(sts);
	sts = pcrSetSweepAtten(handle, PCR_AUTO_ATTEN);
	checkStatus(sts);
    sts = pcrSetSweepStartStop(handle, PCR_MIN_FREQ, PCR_MAX_FREQ);
	checkStatus(sts);
	sts = pcrSetSweepCoupling(handle, rbw, vbw, sweepTime);
	checkStatus(sts);
	sts = pcrSetSweepDetector(handle, PCRSweepDetectorMax, PCRSweepVideoUnitsPower);
	checkStatus(sts);
	sts = pcrSetSweepScale(handle, PCRSweepScaleLog);
	checkStatus(sts);
	sts = pcrSetSweepWindow(handle, PCRSweepWindowFlattop);
	checkStatus(sts);

	// Configure Stream
	double streamRefLevel = -20.0;
	int decimation = 1;

	for(int i = 1; i < CHANNELS; i++) {
		sts = pcrSetChannelShared(handle, i, PCRBoolFalse);
		checkStatus(sts);
		double freqHz = (i + 1) * 1.0e9;
		sts = pcrSetChannelFreq(handle, i, freqHz); // Set each channel to a different frequency
		checkStatus(sts);
	}

	sts = pcrSetStreamDataType(handle, PCRDataType16sc);
	checkStatus(sts);
	sts = pcrSetStreamRefLevel(handle, streamRefLevel);
	checkStatus(sts);
	sts = pcrSetStreamAtten(handle, PCR_AUTO_ATTEN);
	checkStatus(sts);
	sts = pcrSetStreamSampleRate(handle, decimation);
	checkStatus(sts);
	sts = pcrSetStreamBandwidth(handle, PCR_STREAM_MAX_BW);
	checkStatus(sts);

	pcrInitiate(handle);

	// Spin up threads for collecting the data
	std::thread t1(collectSweepData, handle);
	std::thread t2(collectStreamData, handle);
	t1.join(); t2.join();
}
