#include "pcr_api.h"

#include <cassert>
#include <iostream>
#include <Windows.h>

// This example illustrates how to set the PCR4200 into the lower power state.
// This example shows configuring the instrument for multi-channel streaming,
//   querying the power during the stream, stopping the stream, and querying
//   the power before and after putting the device into the low power mode.
// This example shows ~47W consumption when streaming and ~32W when fully
//   in the lower power state. Power consumption will depend on active
//   measurement configuration and frequency.

static void checkStatus(PCRStatus sts)
{
    if(sts < PCRStatusNoError) {
        printf("Status error: %s\n", pcrGetErrorString(sts));
        assert(false);
    } else if(sts > PCRStatusNoError) {
        printf("Status warning: %s\n", pcrGetErrorString(sts));
    }
}

void pcr_example_set_low_power()
{
    PCRStatus sts;
    int handle;

    // Connect the device
    sts = pcrConnectDevice(&handle, PCR_DEFAULT_HOST_ADDR, PCR_DEFAULT_DEVICE_ADDR, PCR_DEFAULT_PORT);
    if(sts != PCRStatusNoError) {
        printf("Error opening device: %s\n", pcrGetErrorString(sts));
        return;
    }

    // Setup a streaming measurement here
    // All 4 channels streaming, independent LO
    int chEnabled[4] = { 1, 1, 1, 1 };
    pcrSetChannelConfig(handle, chEnabled, PCR_SWEEP_CHANNEL_DISABLED);
    for(int i = 0; i < 4; i++) {
        sts = pcrSetChannelShared(handle, i, PCRBoolFalse);
        checkStatus(sts);
        pcrSetChannelFreq(handle, i, 6.0e9);
    }
    sts = pcrSetStreamDataType(handle, PCRDataType32fc);
    checkStatus(sts);
    sts = pcrSetStreamRefLevel(handle, -20.0);
    checkStatus(sts);
    sts = pcrSetStreamAtten(handle, PCR_AUTO_ATTEN);
    checkStatus(sts);
    sts = pcrSetStreamSampleRate(handle, 1);
    checkStatus(sts);
    sts = pcrSetStreamBandwidth(handle, 40.0e6);
    checkStatus(sts);
    sts = pcrSetStreamMode(handle, PCRStreamModeLocal);
    checkStatus(sts);

    // Start the measurements
    sts = pcrInitiate(handle);
    checkStatus(sts);

    // Wait a few seconds, diagnostics only update at 1Hz rate.
    Sleep(2000);

    // Check current power consumption
    float voltage, current;
    pcrGetDeviceDiagnostics(handle, &voltage, &current, nullptr);
    printf("Power State On/Active: %.2f W\n", voltage * current);

    // Now stop the measurement
    pcrAbort(handle);

    // Done with measurements. Remeasure when idle
    pcrAbort(handle);

    // Wait for diagnostics to update, and requery/reprint 
    Sleep(2000);
    sts = pcrGetDeviceDiagnostics(handle, &voltage, &current, nullptr);
    checkStatus(sts);
    printf("Power State On/Idle: %.2f W\n", voltage * current);

    // Now lower power state.
    sts = pcrSetLowPowerState(handle);
    checkStatus(sts);

    // Wait for diagnostics to update, and requery/reprint 
    Sleep(2000);
    sts = pcrGetDeviceDiagnostics(handle, &voltage, &current, nullptr);
    checkStatus(sts);
    printf("Power State Standby: %.2f W\n", voltage * current);

    pcrCloseDevice(handle);
}