#include "pcr_api.h"

// This example illustrates using the built in VSG to calibrate both the phase
// offsets and sample delays between channels. See the Phase Offset and
// Delay Calibration section in the API docs for further description.
// 
// This example extends the code shown in the phase_offset_adjust example by
// adding a second measurement to calculate the subsample delay between channels.
// For a simpler approach that only calibrates the phase offset between channels,
// see pcr_example_phase_offset_adjust. Calibrating out the subsample delays
// between channels corrects phase alignment between channels when measuring
// anywhere within the 40MHz bandwidth of the measurement.
// 
// This example assumes the VSG output is split 4-way into all channels of the PCR.
// 
// 2 CW signals offset from 0Hz are input to all 4 channels. Phase offsets are
// measured at both CW frequency offsets. The phase offset and slope (delay) can be
// calculated for the configured baseband.
// 
// The calculated corrections are applied through the API, and a new measurement
// is made showing the corrected phases.

#include <cassert>
#include <complex>
#include <iostream>
#include <vector>
#include <fstream>

const double PI = 3.14159265;

// Helper function to catch API errors/warnings
static void checkStatus(PCRStatus sts)
{
    if(sts < PCRStatusNoError) {
        printf("Status error: %s\n", pcrGetErrorString(sts));
        assert(false);
    } else if(sts > PCRStatusNoError) {
        printf("Status warning: %s\n", pcrGetErrorString(sts));
    }
}

// Helper function to calculate the average power of a sequence of I/Q samples
static double avgPowerDBM(std::vector<std::complex<float>> &src)
{
    double p = 0.0;
    for(int i = 0; i < src.size(); i++) {
        p += src[i].real() * src[i].real() + src[i].imag() * src[i].imag();
    }

    return 10.0 * log10(p / src.size());
}

// Performs the complex cross correlation between 2 I/Q captures
// Used to measure the phase offset between 2 channels
static std::complex<float> cross_correlation(
    std::vector<std::complex<float>> &s1,
    std::vector<std::complex<float>> &s2)
{
    std::complex<float> result = 0;
    for(int i = 0; i < s1.size(); i++) {
        result += s1[i] * std::conj(s2[i]);
    }
    return result;
}

// Returns a CW signal that can be provided to the VSG. The CW has the
// frequency of freqRatio * 125MHz, where ratio is [-0.5,0.5]. 
// Samples should be a multiple of 16. freqRatio should wrap nicely on the
// length.
static std::vector<std::complex<float>> GetVSGPattern(double freqRatio, int len)
{
    std::vector<std::complex<float>> pattern(len);
    for(int i = 0; i < len; i++) {
        pattern[i].real(cos(freqRatio * 2.0 * PI * i));
        pattern[i].imag(sin(freqRatio * 2.0 * PI * i));
    }
    return pattern;
}

// Main function
void pcr_example_phase_offset_and_delay_adjust()
{
    PCRStatus sts;
    int handle;

    // Correcting all 4 channels
    const int channels = 4;
    // Center frequency used
    const float frequency = 1.0e9;
    // 50MS/s, if decimating, frequency offsets in this example will need to change.
    const int decimation = 1; 
    // Reference level of PCR
    const double refLevel = -10.0;
    // Output power of VSG
    const double vsgLevel = -10.0;
    // CW signal characteristics used for VSG calibration tones
    const double vsgFreqFrac = 1.0 / 16.0;
    const double vsgFreq = vsgFreqFrac * 125.0e6;
    const int vsgPatternLen = 16;

    // Number of I/Q samples to query at a time
    const int N = 1e6;
    // I/Q buffers for receving
    void *ptr[channels];
    std::vector<std::complex<float>> iq[channels];
    for(int i = 0; i < channels; i++) {
        iq[i].resize(N);
        ptr[i] = &iq[i][0];
    }

    // Connect the device
    sts = pcrConnectDevice(&handle, PCR_DEFAULT_HOST_ADDR, PCR_DEFAULT_DEVICE_ADDR, PCR_DEFAULT_PORT);
    if(sts != PCRStatusNoError) {
        printf("Error opening device: %s\n", pcrGetErrorString(sts));
        return;
    }

    // Stores calibration signal measurements
    int chEnabled[channels] = { 1, 1, 1, 1 };
    double avgPower[channels];
    double phaseOffset1[channels] = { 0.0, 0.0, 0.0, 0.0 };
    double phaseOffset2[channels] = { 0.0, 0.0, 0.0, 0.0 };
    double avgPhaseOffset[channels], phaseDelays[channels];

    // Enable all 4 channels for shared LO streaming
    // Initially, set all channels to zero phase offset
    sts = pcrSetChannelConfig(handle, chEnabled, PCR_SWEEP_CHANNEL_DISABLED);
    checkStatus(sts);
    for(int i = 0; i < channels; i++) {
        sts = pcrSetChannelShared(handle, i, PCRBoolTrue);
        checkStatus(sts);
    }

    // Set the shared freuqency, which controls all receive channels on shared
    // LOs and the VSG.
    sts = pcrSetSharedFreq(handle, frequency);
    checkStatus(sts);

    // Configure I/Q stream once up front
    sts = pcrSetStreamDataType(handle, PCRDataType32fc);
    checkStatus(sts);
    sts = pcrSetStreamRefLevel(handle, refLevel);
    checkStatus(sts);
    sts = pcrSetStreamAtten(handle, PCR_AUTO_ATTEN);
    checkStatus(sts);
    sts = pcrSetStreamSampleRate(handle, decimation);
    checkStatus(sts);
    sts = pcrSetStreamBandwidth(handle, 40.0e6);
    checkStatus(sts);
    // We are using the API to retrieve the I/Q samples, and not Vita49 streaming
    sts = pcrSetStreamMode(handle, PCRStreamModeLocal);
    checkStatus(sts);

    { // Measure low side phase offset
        // Configure the VSG to output negative freq tone
        std::vector<std::complex<float>> vsgPattern = GetVSGPattern(-vsgFreqFrac, vsgPatternLen);
        pcrSetVSGPattern(handle, vsgPattern.data(), vsgPattern.size(), PCRDataType32fc);
        checkStatus(sts);
        pcrSetVSGLevel(handle, vsgLevel);
        checkStatus(sts);
        pcrSetVSGEnabled(handle, PCRBoolTrue);
        checkStatus(sts);

        // Start the measurements
        sts = pcrInitiate(handle);
        checkStatus(sts);

        // Now receive I/Q data. We let the device stream for a short time before
        // grabbing data to use for the calibration. See phase settling plots
        // in the product/user manual to see how settling time can affect phase.
        for(int i = 0; i < 10; i++) {
            sts = pcrStreamRecv(handle, (void**)&ptr, N, 0, PCRBoolFalse, 0, 0, 0, 0);
            checkStatus(sts);
        }

        // Measure phase offsets from channel 0
        for(int ch = 0; ch < channels; ch++) {
            avgPower[ch] = avgPowerDBM(iq[ch]);
            std::complex<float> cc = cross_correlation(iq[0], iq[ch]);
            phaseOffset1[ch] = -std::arg(cc);

            printf("Channel %d: Power, %.2f dBm, Phase Offset %.4f rad\n",
                ch+1, avgPower[ch], phaseOffset1[ch]);
        }

        pcrAbort(handle);
    }
    printf("\n");

    { // Measure high side phase offset
        // Configure the VSG
        std::vector<std::complex<float>> vsgPattern = GetVSGPattern(vsgFreqFrac, vsgPatternLen);
        pcrSetVSGPattern(handle, vsgPattern.data(), vsgPattern.size(), PCRDataType32fc);
        checkStatus(sts);
        pcrSetVSGLevel(handle, vsgLevel);
        checkStatus(sts);
        pcrSetVSGEnabled(handle, PCRBoolTrue);
        checkStatus(sts);

        // Start the measurements
        sts = pcrInitiate(handle);
        checkStatus(sts);

        // Now receive I/Q data. We let the device stream for a short time before
        // grabbing data to use for the calibration. See phase settling plots
        // in the product/user manual to see how settling time can affect phase.
        for(int i = 0; i < 10; i++) {
            sts = pcrStreamRecv(handle, (void**)&ptr, N, 0, PCRBoolFalse, 0, 0, 0, 0);
            checkStatus(sts);
        }

        // Cross correlation approach
        for(int ch = 0; ch < channels; ch++) {
            avgPower[ch] = avgPowerDBM(iq[ch]);
            std::complex<float> cc = cross_correlation(iq[0], iq[ch]);
            phaseOffset2[ch] = -std::arg(cc);

            printf("Channel %d: Power, %.2f dBm, Phase Offset %.4f rad\n",
                ch+1, avgPower[ch], phaseOffset2[ch]);
        }

        pcrAbort(handle);
    }
    printf("\n");

    // Calculate average phase offset and phase delay
    for(int i = 0; i < channels; i++) {
        double phaseDelta = phaseOffset2[i] - phaseOffset1[i];
        if(phaseDelta > PI) {
            phaseOffset2[i] -= 2.0 * PI;
        } else if(phaseDelta < -PI) {
            phaseOffset2[i] += 2.0 * PI;
        }

        avgPhaseOffset[i] = (phaseOffset1[i] + phaseOffset2[i]) / 2.0;

        // Convert to offset per Hz
        double offset = (phaseOffset2[i] - phaseOffset1[i]) / (2.0 * vsgFreq);
        phaseDelays[i] = offset / (2.0*PI);
    }

    // Now apply the calculated phase offset
    for(int i = 0; i < channels; i++) {
        printf("Channel %d: Offset %.5f Delay %f ns\n", 
            i+1, avgPhaseOffset[i], phaseDelays[i] * 1.0e9);

        sts = pcrSetChannelPhaseOffset(handle, i, avgPhaseOffset[i]);
        checkStatus(sts);
        sts = pcrSetChannelDelay(handle, i, phaseDelays[i]);
        checkStatus(sts);
    }
    printf("\n");

    // Now measure the phase offset across the I/Q bandwidth
    std::ofstream file("phase_offset_vs_freq.csv");

    for(int i = -10; i <= 10; i++) {
        const int vsgPatternLen = 64;
        const double f = (double)i / 64.0;

        std::vector<std::complex<float>> vsgPattern = GetVSGPattern(f, 64);
        pcrSetVSGPattern(handle, vsgPattern.data(), vsgPatternLen, PCRDataType32fc);
        checkStatus(sts);
        pcrSetVSGEnabled(handle, PCRBoolTrue);
        checkStatus(sts);

        // Start stream
        sts = pcrInitiate(handle);
        checkStatus(sts);

        // Collect some data, throw away initial samples
        for(int i = 0; i < 2; i++) {
            sts = pcrStreamRecv(handle, (void**)&ptr, N, 0, PCRBoolFalse, 0, 0, 0, 0);
            checkStatus(sts);
        }

        printf("VSG Freq Offset %.2f MHz\n", f * 125.0);

        // Cross correlation approach
        file << f << ",";
        for(int ch = 0; ch < channels; ch++) {
            avgPower[ch] = avgPowerDBM(iq[ch]);
            std::complex<float> cc = cross_correlation(iq[0], iq[ch]);
            double correctedPhaseOffset = -std::arg(cc);

            printf("Channel %d: Power, %.2f dBm, Phase Offset %.4f rad\n",
                ch+1, avgPower[ch], correctedPhaseOffset);
            file << correctedPhaseOffset << ",";
        }
        file << std::endl;

        pcrAbort(handle);
        printf("\n");
    }

    file.close();

    // Done, close device
    pcrCloseDevice(handle);
}