#include "pcr_api.h"

// This example illustrates creating a multi system using up to N PCR4200s.
// The process of connecting and configuring this system is illustrated.   
// Once configured, the N*4 channel system is interfaced using a single handle
// and can be configured through all the standard API routines.

#include <cassert>
#include <complex>
#include <iostream>
#include <vector>
#include <Windows.h>

static void checkStatus(PCRStatus sts)
{
    if(sts < PCRStatusNoError) {
        printf("Status error: %s\n", pcrGetErrorString(sts));
        assert(false);
    } else if(sts > PCRStatusNoError) {
        printf("Status warning: %s\n", pcrGetErrorString(sts));
    }
}

static double avgPowerDBM(std::vector<std::complex<float>> &src)
{
    double p = 0.0;
    for(int i = 0; i < src.size(); i++) {
        p += src[i].real() * src[i].real() + src[i].imag() * src[i].imag();
    }

    return 10.0 * log10(p / src.size());
}

void pcr_example_multi_receiver_1()
{
    PCRStatus sts;
    int handle;

    // Configure the network settings to match your system
    const char *hostAddr[4] = {
        "192.168.2.3",
        "192.168.3.3",
        "192.168.4.3",
        "192.168.5.3"
    };

    const char *deviceAddr[4] = {
        "192.168.2.10",
        "192.168.3.10",
        "192.168.4.10",
        "192.168.5.10"
    };

    int devicePort[4] = {
        PCR_DEFAULT_PORT,
        PCR_DEFAULT_PORT,
        PCR_DEFAULT_PORT,
        PCR_DEFAULT_PORT
    };

    const int DEVICES = 4;
    const int TOTAL_CHANNELS = DEVICES * 4;
    const double frequency = 1.0e9;
    const int decimation = 4;
    const double sampleRate = PCR_STREAM_SR / decimation;
    const double refLevel = -10.0;
    const double bandwidth = 40.0e6 / decimation;

    printf("Connecting primary device\n");
    sts = pcrConnectDevice(&handle, hostAddr[0], deviceAddr[0], devicePort[0]);
    checkStatus(sts);
    // Pair the other device
    for(int i = 1; i < DEVICES; i++) {
        printf("Pairing another device\n");
        sts = pcrPairDevice(handle, hostAddr[i], deviceAddr[i], devicePort[i]);
        checkStatus(sts);
    }
    printf("Connecting/Pairing complete\n");

    // These functions set the local stream port. The streaming port is
    // different than the port used to connect the device. By default
    // the stream port will be 4991 on both devices. If on the same subnet
    // this will be a problem. This function allows you to change that.
    // In this example, the initial configuration was for 2 separate subnets,
    // and these functions are technically needed.
    //sts = pcrSetStreamLocalAddr(handle, 0, PCR_DEFAULT_STREAM_PORT);
    //checkStatus(sts);
    //sts = pcrSetStreamLocalAddr(handle, 1, PCR_DEFAULT_STREAM_PORT+1);
    //checkStatus(sts);

    // Confirm we have an 4*N channel system
    int channelCount = 0;
    sts = pcrGetChannelCount(handle, &channelCount);
    checkStatus(sts);
    assert(channelCount == TOTAL_CHANNELS);

    std::vector<int> chEnabled(TOTAL_CHANNELS, 1);

    // Enable all 4 channels for shared LO streaming
    // Initially, set all channels to zero phase offset
    sts = pcrSetChannelConfig(handle, chEnabled.data(), PCR_SWEEP_CHANNEL_DISABLED);
    checkStatus(sts);
    for(int i = 0; i < channelCount; i++) {
        sts = pcrSetChannelShared(handle, i, PCRBoolTrue);
        checkStatus(sts);
    }

    // VSG uses the shared frequency. So this function will not only control
    //  any receive channels using the shared frequency, but also the VSG output frequency.
    // For this example, the recieve channel will use the shared frequency.
    sts = pcrSetSharedFreq(handle, frequency);
    checkStatus(sts);

    // Configure I/Q stream
    sts = pcrSetStreamDataType(handle, PCRDataType32fc);
    checkStatus(sts);
    sts = pcrSetStreamRefLevel(handle, refLevel);
    checkStatus(sts);
    sts = pcrSetStreamAtten(handle, PCR_AUTO_ATTEN);
    checkStatus(sts);
    sts = pcrSetStreamSampleRate(handle, decimation);
    checkStatus(sts);
    sts = pcrSetStreamBandwidth(handle, bandwidth);
    checkStatus(sts);
    // We are using the API to retrieve the I/Q samples, and not Vita49 streaming
    sts = pcrSetStreamMode(handle, PCRStreamModeLocal);
    checkStatus(sts);

    // Pattern is provided as interleaved I/Q samples
    // The pattern must be a multiple of 16 I/Q samples
    // Fixed DC/CW signal
    const int vsgPatternLen = 16;
    std::vector<std::complex<float>> vsgPattern(vsgPatternLen);
    for(int i = 0; i < vsgPatternLen; i++) {
        vsgPattern[i].real(1.0);
        vsgPattern[i].imag(0.0);
    }

    // Configure the VSG
    pcrSetVSGPattern(handle, vsgPattern.data(), vsgPatternLen, PCRDataType32fc);
    checkStatus(sts);
    pcrSetVSGLevel(handle, refLevel);
    checkStatus(sts);
    pcrSetVSGEnabled(handle, PCRBoolTrue);
    checkStatus(sts);

    for(double f = 100.0e6; f < 5.7e9; f += 100.0e6) {
        pcrSetSharedFreq(handle, f);
        sts = pcrInitiate(handle);
        checkStatus(sts);
        Sleep(250);
        pcrAbort(handle);
    }

    // Start the measurements
    sts = pcrInitiate(handle);
    checkStatus(sts);

    // Number of samples to query at a time
    int N = 1e6;
    void *ptr[TOTAL_CHANNELS];
    std::vector<std::complex<float>> iq[TOTAL_CHANNELS];

    for(int i = 0; i < TOTAL_CHANNELS; i++) {
        iq[i].resize(N);
        ptr[i] = &iq[i][0];
    }

    // Collect some data.
    int iter = 0;
    while(true) {
        int samplesRemaining;
        sts = pcrStreamRecv(handle, (void**)&ptr, N, 0, PCRBoolFalse, 0, 
            &samplesRemaining, 0, 0);
        checkStatus(sts);

        // Measure phase offsets between channels
        if(++iter % 50 == 0) {
            printf("Samples remaining: %d\n", samplesRemaining);
            for(int ch = 0; ch < TOTAL_CHANNELS; ch++) {
                double avgPower = avgPowerDBM(iq[ch]);
                printf("Channel %d: Power, %.2f dBm\n", ch+1, avgPower);
            }
            printf("\n");
        }
    }

    pcrAbort(handle);
    pcrCloseDevice(handle);
}