#include "pcr_api.h"

#include <iostream>
#include <cassert>
#include <vector>
#include <complex>

#include <Windows.h>

void pcr_example_iq_stream_with_timestamping()
{
    PCRStatus sts;
    int handle;
    sts = pcrConnectDevice(&handle, PCR_DEFAULT_HOST_ADDR, PCR_DEFAULT_DEVICE_ADDR, PCR_DEFAULT_PORT);
    assert(sts == PCRStatusNoError);
    printf("Connected\n");

    sts == pcrSetGPSTimebaseUpdate(handle, PCRBoolTrue);
    assert(sts == PCRStatusNoError);

    // Wait until GPS is locked
    printf("Waiting for GPS locked.\n");
    PCRGPSState state = PCRGPSStateNotPresent;
    do {
        pcrGetGPSState(handle, &state);
        Sleep(1000);
    } while(state != PCRGPSStateLocked);

    printf("GPS locked\n");

    // Enable just one channel
    const int chEnabled[4] = { 1, 0, 0, 0 };
    const int decimation = 1;

    // Configure Stream
    sts = pcrSetChannelConfig(handle, chEnabled, PCR_SWEEP_CHANNEL_DISABLED);
    assert(sts == PCRStatusNoError);
    sts = pcrSetChannelShared(handle, 0, PCRBoolTrue);
    assert(sts == PCRStatusNoError);
    sts = pcrSetSharedFreq(handle, 1000.0e6);
    assert(sts == PCRStatusNoError);
    sts = pcrSetVSGEnabled(handle, PCRBoolFalse);
    assert(sts == PCRStatusNoError);
    sts = pcrSetStreamDataType(handle, PCRDataType32fc);
    assert(sts == PCRStatusNoError);
    sts = pcrSetStreamRefLevel(handle, -20.0);
    assert(sts == PCRStatusNoError);
    sts = pcrSetStreamAtten(handle, PCR_AUTO_ATTEN);
    assert(sts == PCRStatusNoError);
    sts = pcrSetStreamSampleRate(handle, decimation);
    assert(sts == PCRStatusNoError);
    sts = pcrSetStreamBandwidth(handle, PCR_STREAM_MAX_BW / (double)decimation);
    assert(sts == PCRStatusNoError);
    sts = pcrSetStreamTriggerEdge(handle, PCRTriggerEdgeRising);
    assert(sts == PCRStatusNoError);
    sts = pcrInitiate(handle);
    assert(sts == PCRStatusNoError);

    // Allocate memory for I/Q data
    const int len = PCR_STREAM_SR / decimation;
    std::vector<std::complex<float>> buf(len);
    void *ptr[1];
    ptr[0] = &buf[0];
    int64_t nsSinceEpoch = 0;
    std::vector<double> triggers;
    std::vector<uint64_t> triggerTimes;
    triggers.resize(10);

    const int64_t nsPerSample = 20 * decimation;

    // Capture 1 second of data at a time, print the timestamps of 
    //   any triggers found during that capture.
    while(true) {
        pcrGetGPSState(handle, &state);
        sts = pcrStreamRecv(handle, (void**)ptr, len, &nsSinceEpoch, 
            PCRBoolTrue, 0, 0, triggers.data(), triggers.size());

        if(sts != PCRStatusNoError) {
            printf("Status: %s\n", pcrGetErrorString(sts));
        }

        for(int i = 0; i < triggers.size(); i++) {
            int64_t triggerOffset = triggers[i] * nsPerSample;
            int64_t triggerTime = nsSinceEpoch + triggerOffset;
            if(triggers[i] != -1.0) {
                printf("GPS %s, ", (state == PCRGPSStateDisciplined) ?
                    "Disciplined" : "Locked");
                printf("Tigger Time ");
                printf("%lld.", triggerTime / 1'000'000'000);
                printf("%09lld seconds since epoch\n", triggerTime % 1'000'000'000);
            }
        }
    }

    pcrAbort(handle);
    pcrCloseDevice(handle);
}
