#include "pcr_api.h"

#include <iostream>
#include <cassert>
#include <vector>
#include <complex>

// This example configures a single I/Q streaming channel using the
// shared LO. Also shown is how to query I/Q data.

static void checkStatus(PCRStatus sts)
{
    if(sts < PCRStatusNoError) {
        printf("Status error: %s\n", pcrGetErrorString(sts));
        assert(false);
    } else if(sts > PCRStatusNoError) {
        printf("Status warning: %s\n", pcrGetErrorString(sts));
    }
}

void pcr_example_iq_simple()
{
    PCRStatus sts;
    int handle;

    // Decimation should be a power of 2
    const int decimation = 16;
    const double expectedSampleRate = PCR_STREAM_SR / decimation;

    // Connect the device
    sts = pcrConnectDevice(&handle, PCR_DEFAULT_HOST_ADDR, PCR_DEFAULT_DEVICE_ADDR, PCR_DEFAULT_PORT);
    if(sts != PCRStatusNoError) {
        printf("Error opening device: %s\n", pcrGetErrorString(sts));
        return;
    }

    // Enable one stream channel, between [0,3]
    const int activeChannel = 0;
    int chEnabled[4] = { 0, 0, 0, 0 };
    chEnabled[activeChannel] = 1;

    // Set channel configuration
    sts = pcrSetChannelConfig(handle, chEnabled, PCR_SWEEP_CHANNEL_DISABLED);
    checkStatus(sts);
    sts = pcrSetChannelShared(handle, activeChannel, PCRBoolTrue);
    checkStatus(sts);
    sts = pcrSetSharedFreq(handle, 1.0e9);
    checkStatus(sts);

    // Configure I/Q stream
    sts = pcrSetStreamDataType(handle, PCRDataType32fc);
    checkStatus(sts);
    sts = pcrSetStreamRefLevel(handle, -20.0);
    checkStatus(sts);
    sts = pcrSetStreamAtten(handle, PCR_AUTO_ATTEN);
    checkStatus(sts);
    sts = pcrSetStreamSampleRate(handle, decimation);
    checkStatus(sts);
    sts = pcrSetStreamBandwidth(handle, 40.0e6);
    checkStatus(sts);
    // We are using the API to retrieve the I/Q samples, and not Vita49 streaming
    sts = pcrSetStreamMode(handle, PCRStreamModeLocal);
    checkStatus(sts);

    // Start the measurements
    sts = pcrInitiate(handle);
    checkStatus(sts);

    // Query the stream parameters
    int channelCount;
    double sampleRate, bandwidth;
    sts = pcrStreamParameters(handle, &channelCount, &sampleRate, &bandwidth);
    checkStatus(sts);

    assert(channelCount == 1);
    assert(sampleRate == expectedSampleRate);

    // Number of samples to query at a time
    int N = 1e6;
    std::vector<std::complex<float>> iq(N);

    // Collect 1 second worth of data
    for(int i = 0; i < 50; i++) {
        std::complex<float> *ptr = iq.data();
        sts = pcrStreamRecv(handle, (void**)&ptr, N, 0, PCRBoolFalse, 0, 0, 0, 0);
        checkStatus(sts);

        // Do something with the data here...
    }

    // Done, close device
    pcrCloseDevice(handle);
}